<?php

namespace App\Services;

use App\Enums\CouponDiscountTypeEnum;
use App\Exceptions\CouponExpiredException;
use App\Exceptions\InvalidBusinessCoupon;
use App\Exceptions\InvalidCouponExpiryDate;
use App\Exceptions\InvalidCouponStartDate;
use App\Models\Coupon;
use App\Models\CouponProductExclusion;
use App\Models\Product;
use App\Models\User;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;


trait CouponService
{

    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Coupon::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Check if is percentage card discount
     *
     * @return bool
     */
    public function isPercentageCartDiscount(): bool
    {
        return $this->{"discount_type"} == CouponDiscountTypeEnum::percentageDiscount->value;
    }

    /**
     * Check if is fixed card discount
     *
     * @return bool
     */
    public function isFixedCartDiscount(): bool
    {
        return $this->{"discount_type"} == CouponDiscountTypeEnum::fixedCardDiscount->value;
    }

    /**
     * Check if is product discount by percentage
     *
     * @return bool
     */
    public function isProductDiscountByPercentage(): bool
    {
        return $this->{"discount_type"} == CouponDiscountTypeEnum::productDiscount->value && $this->{"percentage_discount"} != null;
    }

    /**
     * CHeck if is product discount by fixed amount
     *
     * @return bool
     */
    public function isProductDiscountByFixedAmount(): bool
    {
        return $this->{"discount_type"} == CouponDiscountTypeEnum::productDiscount->value && $this->{"fixed_card_discount_amount"} != null;
    }

    /**
     * Get total number of percentage discounts
     * @param int $businessId
     * @return int
     */
    public static function getTotalNumberOfPercentageDiscounts(int $businessId): int
    {
        return Coupon::query()->where('business_id', $businessId)->where('discount_type', CouponDiscountTypeEnum::percentageDiscount->value)->count();
    }

    /**
     * Get total number of fixed card discounts
     *
     * @param int $businessId
     * @return int
     */
    public static function getTotalNumberOfFixedCardDiscounts(int $businessId): int
    {
        return Coupon::query()->where('business_id', $businessId)->where('discount_type', CouponDiscountTypeEnum::fixedCardDiscount->value)->count();
    }

    /**Get total number of product discounts
     *
     * @param int $businessId
     * @return int
     */
    public static function getTotalNumberOfProductDiscounts(int $businessId): int
    {
        return Coupon::query()->where('business_id', $businessId)->where('discount_type', CouponDiscountTypeEnum::productDiscount->value)->count();
    }

    /**
     * Invalid coupon expiry date
     *
     * @param string $expiryDate
     * @throws InvalidCouponExpiryDate
     */
    public static function isValidExpiryDate(string $expiryDate): void
    {
        $currentDate = Carbon::now();
        if ($currentDate->gt($expiryDate)) {
            throw  new  InvalidCouponExpiryDate(__("errors.invalid_coupon_expiry_date"));
        }
    }

    /**
     * Invalid coupon start date
     *
     * @param string $startDate
     * @param string $expiryDate
     * @throws InvalidCouponStartDate
     */
    public static function isValidStartDate(string $startDate, string $expiryDate) {
        $start = Carbon::createFromFormat('Y-m-d H:i:s', $startDate);
        $expiry = Carbon::createFromFormat('Y-m-d H:i:s', $expiryDate);
        if($start->gt($expiry)) {
            throw new InvalidCouponStartDate(__("errors.invalid_coupon_start_date"));
        }
    }

    /**
     * Coupon expired
     *
     * @throws CouponExpiredException
     */
    public function hasExpired() {
        $currentDate = Carbon::now();
        if ($currentDate->gt($this->{"expiring_date"})) {
            throw new CouponExpiredException(__("errors.coupon_expired"));
        }
    }

   /**
    * Invalid business coupon exception
    *
    * @param int $ProductBusinessId
    * @throws InvalidBusinessCoupon
    */
   public function isValidBusinessCoupon(int $ProductBusinessId) {
      if($this->{"business_id"} != $ProductBusinessId) {
         throw new InvalidBusinessCoupon(__("errors.invalid_business_coupon"));
      }
   }

    /**
     * Create coupon
     *
     * @param array $data
     * @return Builder|Model
     * @throws InvalidCouponExpiryDate
     * @throws Exception
     */
    public static function create(array $data): Model|Builder
    {
        try {
            /** @var User $user */
            $user = auth()->user();
            $business = $user->{'business'};
            $wallet = $user->{'wallet'};

            //Check expiry date is valid
            self::isValidExpiryDate($data["expiring_date"]);

            //Check start date is not greater than expiry date
            if(isset($data["product_discount_start_date"])) {
                self::isValidStartDate($data["product_discount_start_date"], $data["expiring_date"]);
            }

            DB::beginTransaction();

            /** @var Coupon $coupon */
            $coupon = self::store([
                "designation" => $data["designation"],
                "description" => $data["description"],
                "discount_type" => $data["discount_type"],
                "percentage_discount" => $data["percentage_discount"] ?? null,
                "fixed_card_discount_amount" => $data["fixed_card_discount_amount"] ?? null,
                "product_discount_start_date" =>  convert_datetime_to_utc($data["product_discount_start_date"], $data[config('torryme.constants.time_zone_key')]) ?? null,
                "expiring_date" => convert_datetime_to_utc($data["expiring_date"], $data[config('torryme.constants.time_zone_key')]),
                "minimum_amount_to_spend" => $data["minimum_amount_to_spend"] ?? null,
                "maximum_amount_to_spend" => $data["maximum_amount_to_spend"] ?? null,
                "fixed_card_discount_currency_id" => $data["fixed_card_discount_amount"] ?? null ? $wallet->{"currency_id"} : null,
                "minimum_amount_to_spend_currency_id" => $data["minimum_amount_to_spend"] != null ? $wallet->{"currency_id"} : null,
                "maximum_amount_to_spend_currency_id" => $data["maximum_amount_to_spend"] != null ? $wallet->{"currency_id"} : null,
                "product_id" => $data["product_id"] ?? null,
                "business_id" => $business->{"id"},
            ]);

            //Get first word of discount type
            $words = explode(" ", $data["discount_type"]);
            $discountTypeInitials = "";
            foreach ($words as $w) {
                $discountTypeInitials .= $w[0];
            }

            //Update coupon code
            $coupon->updateService([
                "code" => generate_coupon_code($discountTypeInitials, $coupon->{"id"})
            ]);

            if (isset($data["excluded_product_ids"]) && sizeof($data["excluded_product_ids"]) > 0) {
                foreach ($data["excluded_product_ids"] as $productId) {
                    CouponProductExclusion::store([
                        "coupon_id" => $coupon->{"id"},
                        "product_id" => $productId,
                    ]);
                }
            }

            DB::commit();

            $result = $coupon->refresh();
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'CouponService::create');
            DB::rollBack();

            throw  $exception;
        }

        return $result;
    }

    /**
     * Get coupon discount type statistics
     *
     * @return array
     */
    public static function statistics(): array
    {
        /** @var User $user */
        $user = auth()->user();
        $business = $user->{'business'};

        return [
            "total_number_of_percentage_discounts" => self::getTotalNumberOfPercentageDiscounts($business->{"id"}),
            "total_number_of_fixed_card_discounts" => self::getTotalNumberOfFixedCardDiscounts($business->{"id"}),
            "total_number_of_product_discounts" => self::getTotalNumberOfProductDiscounts($business->{"id"}),
        ];
    }

    /**
     * Get discount type
     *
     * @param string|null $discountType
     * @param string|null $businessId
     * @param int|null $pageNumber
     * @return LengthAwarePaginator
     * @throws Exception
     */
    public static function discountTypes(string $discountType = null, string $businessId = null, int $pageNumber = null): LengthAwarePaginator
    {
        $user = auth()->user();
        $business = $user->{'business'};
        $date = date('Y-m-d H:i:s');

        $queryBuilder = Coupon::query();

        if($businessId == null && $business != null) {
            if ($discountType == CouponDiscountTypeEnum::percentageDiscount->value) {
                $queryBuilder->where('business_id', $business->{"id"})->where('discount_type', CouponDiscountTypeEnum::percentageDiscount->value);
            } elseif ($discountType == CouponDiscountTypeEnum::fixedCardDiscount->value) {
                $queryBuilder->where('business_id', $business->{"id"})->where('discount_type', CouponDiscountTypeEnum::fixedCardDiscount->value);
            } elseif ($discountType == CouponDiscountTypeEnum::productDiscount->value) {
                $queryBuilder->where('business_id', $business->{"id"})->where('discount_type', CouponDiscountTypeEnum::productDiscount->value);
            } else {
                $queryBuilder->where('business_id', $business->{"id"});
            }
        } else {
            if($business != null && $businessId == $business->{"id"}) {
                if ($discountType == CouponDiscountTypeEnum::percentageDiscount->value) {
                    $queryBuilder->where('business_id', $businessId)->where('discount_type', CouponDiscountTypeEnum::percentageDiscount->value);
                } elseif ($discountType == CouponDiscountTypeEnum::fixedCardDiscount->value) {
                    $queryBuilder->where('business_id', $businessId)->where('discount_type', CouponDiscountTypeEnum::fixedCardDiscount->value);
                } elseif ($discountType == CouponDiscountTypeEnum::productDiscount->value) {
                    $queryBuilder->where('business_id', $businessId)->where('discount_type', CouponDiscountTypeEnum::productDiscount->value);
                } else {
                    $queryBuilder->where('business_id', $businessId);
                }
            } else {
                if ($discountType == CouponDiscountTypeEnum::percentageDiscount->value) {
                    $queryBuilder->where('business_id', $businessId)->where('discount_type', CouponDiscountTypeEnum::percentageDiscount->value)->whereDate('product_discount_start_date', '<', $date)->whereDate('expiring_date', '>', $date);
                } elseif ($discountType == CouponDiscountTypeEnum::fixedCardDiscount->value) {
                    $queryBuilder->where('business_id', $businessId)->where('discount_type', CouponDiscountTypeEnum::fixedCardDiscount->value)->whereDate('product_discount_start_date', '<', $date)->whereDate('expiring_date', '>', $date);
                } elseif ($discountType == CouponDiscountTypeEnum::productDiscount->value) {
                    $queryBuilder->where('business_id', $businessId)->where('discount_type', CouponDiscountTypeEnum::productDiscount->value)->whereDate('product_discount_start_date', '<', $date)->whereDate('expiring_date', '>', $date);
                } else {
                    $queryBuilder->where('business_id', $businessId)->whereDate('product_discount_start_date', '<', $date)->whereDate('expiring_date', '>', $date);
                }
            }
        }

        return
            $queryBuilder
                ->orderByDesc('created_at')
                ->paginate(
                    perPage: config("torryme.constants.items_per_page"),
                    page: $pageNumber
                );
    }

    /**
     * Compute discount types from paginator
     *
     * @param LengthAwarePaginator $discountTypes
     * @return array
     */
    public static function computeCouponTypesFromPaginator(LengthAwarePaginator $discountTypes): array
    {
        $DiscountTypeArray = array();
        foreach ($discountTypes as $discountType) {
            $DiscountTypeArray[] = $discountType->computeSerialization() ;
        }

        return default_paginator_format(
            $discountTypes->lastPage(),
            $discountTypes->total(),
            $discountTypes->currentPage(),
            "discount_types",
            $DiscountTypeArray,
        );
    }

    /**
     * Find by coupon code
     *
     * @param string $code
     * @return Model|Builder|null
     */
    public static function findByCode(string $code): Model|Builder|null
    {
        return Coupon::query()->where('code', $code)->first();
    }

    /**
     * Compute for serialization
     *
     * @return Coupon
     */
    public function computeSerialization(): Coupon
    {
        $coupon = $this;

        if($coupon->{'discount_type'} === CouponDiscountTypeEnum::productDiscount->value) {
            /** @var Product $product */
            $product = $coupon->{'product'};
            unset($coupon->{'product'});

            $coupon['product'] = $product->computeDetailsForOrderDetails();
        }

        if($coupon->{'fixed_card_discount_amount'} !== null) {
            $coupon = $coupon->load('fixedCardDiscountCurrency');
        }

        if($coupon->{'minimum_amount_to_spend'} !== null) {
            $coupon = $coupon->load('minimumAmountToSpendCurrency');
        }

        if($coupon->{'maximum_amount_to_spend'} !== null) {
            $coupon = $coupon->load('maximumAmountToSpendCurrency');
        }

        return $coupon;
    }
}
