<?php

namespace App\Services;

use App\Models\Country;
use Giggsey\Locale\Locale;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use libphonenumber\NumberParseException;
use libphonenumber\PhoneNumberUtil;
use Propaganistas\LaravelPhone\PhoneNumber;

trait CountryService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      if (isset($data['country_code'])) {
         $data['country_code'] = mb_strtoupper($data['country_code']);
      }

      return Country::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      if (isset($data['country_code'])) {
         $data['country_code'] = mb_strtoupper($data['country_code']);
      }

      return tap($this)->update($data);
   }

   /**
    * Parse phone number and get country info
    *
    * @param string $phoneNumber
    * @return Country|null
    */
   public static function getCountryInfo(string $phoneNumber): ?Country
   {
      $phoneUtil = PhoneNumberUtil::getInstance();
      try {
         $phoneNumberProto = $phoneUtil->parse(
            remove_space(trim($phoneNumber))
         );

         if ($phoneNumberProto) {
            $countryCode = (new PhoneNumber($phoneNumber))->getCountry();
            $countryName = Locale::getDisplayRegion(
               sprintf('-%s', strtoupper($countryCode)),
               config('torryme.locales_keys.en')
            );

            // Find country by dial code
            $country = new Country();
            $country->{'designation'} = $countryName;
            $country->{'dial_code'} = '+' . $phoneNumberProto->getCountryCode();
            $country->{'country_code'} = mb_strtoupper($countryCode);

            return $country;
         }
      } catch (NumberParseException $e) {
         log_debug($e, 'CountryService::getCountryInfo');
      }

      return null;
   }

   /**
    * Get country info using country code
    *
    * @param string $code
    * @return Country|null
    */
   public static function getCountryInfoByCode(string $code): ?Country
   {
      $phoneUtil = PhoneNumberUtil::getInstance();

      try {
         $dialCode = $phoneUtil->getCountryCodeForRegion($code);
         $countryName = Locale::getDisplayRegion(
            sprintf('-%s', strtoupper($code)),
            config('torryme.locales_keys.en')
         );

         // Find country by dial code
         $country = new Country();
         $country->{'designation'} = $countryName;
         $country->{'dial_code'} = '+' . $dialCode;
         $country->{'country_code'} = mb_strtoupper($code);

         return $country;
      } catch (\Exception $e) {
         log_debug($e, 'CountryService::getCountryInfoByCode');
      }

      return null;
   }

   /**
    * Handle platform country
    *
    * @param string $phoneNumber
    * @return Builder|Model|null
    */
   public static function handleAllCountries(string $phoneNumber): Model|Builder|null
   {
      $countrySchema = self::getCountryInfo($phoneNumber);

      if ($countrySchema !== null) {
         $country = Country::findByCountryCode(
            $countrySchema->{'country_code'}
         );

         if ($country !== null) {
            return $country;
         }

         return Country::store(array(
            'designation' => $countrySchema->{'designation'},
            'dial_code' => $countrySchema->{'dial_code'},
            'country_code' => $countrySchema->{'country_code'},
         ));
      }

      return null;
   }

   /**
    * Create or update country
    *
    * @param string $code
    * @return Model|Builder|null
    */
   public static function createOrUpdate(string $code): Model|Builder|null
   {
      $country = Country::findByCountryCode($code);
      if ($country === null) {
         $country = self::getCountryInfoByCode($code);

         if ($country !== null) {
            return Country::store(array(
               'designation' => $country->{'designation'},
               'dial_code' => $country->{'dial_code'},
               'country_code' => $country->{'country_code'},
            ));
         }

         return null;
      }

      return $country;
   }

   /**
    * Find by country code
    *
    * @param ?string $countryCode
    * @return Builder|Model|null
    */
   public static function findByCountryCode(?string $countryCode): Model|Builder|null
   {
      if (!filled($countryCode)) {
         return null;
      }

      return Country::query()->where('country_code', mb_strtoupper($countryCode))->first();
   }

   /**
    * All countries
    *
    * @return Model|Builder|null
    */
   public static function allCountries(): Model|Builder|null
   {
      return Country::query();
   }
}
