<?php

namespace App\Services;

use App\Enums\CommunitySuspensionDurationEnum;
use App\Enums\GenericStatusEnum;
use App\Exceptions\InvalidSuspensionAction;
use App\Models\CommunitySubscriber;
use App\Models\CommunitySuspension;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Mockery\Exception;

trait CommunitySuspensionService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return CommunitySuspension::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    * Find suspension
    *
    * @param int $communityId
    * @param int $userId
    * @return Model|Builder|null
    */
   public static function findSuspension(int $userId, int $communityId): Model|Builder|null
   {
      return CommunitySuspension::query()->where("user_id", $userId)->where("community_id", $communityId)->first();
   }

   /**
    * Add community suspension
    *
    * @param array $data
    * @return Builder|Model|null
    */
   public static function addSuspension(array $data): Model|Builder|null
   {
      try {
         if($data["suspension_duration_type"] == CommunitySuspensionDurationEnum::custom->value) {
            $suspendedUntil = convert_datetime_to_utc($data["suspension_duration"], $data[config('torryme.constants.time_zone_key')]);
         } else {
            $currentDate = Carbon::now();
            $currenDateInUTC = convert_datetime_to_utc($currentDate, $data[config('torryme.constants.time_zone_key')]);

            $suspendedUntil = match ($data["suspension_duration_type"]) {
               CommunitySuspensionDurationEnum::twentyFourHours->value => $currenDateInUTC->addHours(config("torryme.community_suspension_duration.24_hours")),
               CommunitySuspensionDurationEnum::sevenDays->value => $currenDateInUTC->addDays(config("torryme.community_suspension_duration.7_days")),
               CommunitySuspensionDurationEnum::fourteenDays->value => $currenDateInUTC->addDays(config("torryme.community_suspension_duration.14_days")),
               CommunitySuspensionDurationEnum::thirtyDays->value => $currenDateInUTC->addDays(config("torryme.community_suspension_duration.30_days")),
               CommunitySuspensionDurationEnum::mintyDays->value => $currenDateInUTC->addDays(config("torryme.community_suspension_duration.90_days")),
               CommunitySuspensionDurationEnum::twelveMonths->value => $currenDateInUTC->addMonths(config("torryme.community_suspension_duration.12_months")),
               default => $currentDate,
            };
         }

         /** @var CommunitySuspension $suspensionExist */
         $suspensionExist = self::findSuspension($data["user_id"], $data["community_id"]);
         if($suspensionExist !== null) {
            $communitySuspension = $suspensionExist->updateService([
               "suspended_until" => $suspendedUntil,
            ]);
         } else {
            $communitySuspension = self::store([
               "suspended_until" => $suspendedUntil,
               "community_id" => $data["community_id"],
               "user_id" => $data["user_id"],
            ]);
         }

      } catch (Exception $exception) {
         log_debug($exception, 'CommunitySuspensionService::addSuspension');
         $communitySuspension = null;
      }

      return $communitySuspension;
   }

   /**
    * Remove community suspension
    *
    * @param array $data
    * @return CommunitySuspension
    * @throws InvalidSuspensionAction
    */
   public function removeSuspension(array $data): CommunitySuspension
   {
      try {
         $suspension = $this;

         /** @var User $user */
         $user = auth()->user();

         // Check if action is performed by community admin
         $isCommunityAdmin = CommunitySubscriber::checkIsCommunityAdminOrAlreadySubscriber(GenericStatusEnum::enable->value, $user->{"id"}, $data["community_id"]);

         if ($isCommunityAdmin == null) {
            throw new InvalidSuspensionAction(__("errors.invalid_subscriber_action"));
         }

         //Delete community suspension
         $suspension->delete();

      } catch (\Exception $exception) {
         log_debug($exception, "CommunitySuspensionService::removeSuspension");
         throw $exception;
      }

      return $suspension;
   }

}
