<?php

namespace App\Services;

use App\Enums\DocumentTypeEnum;
use App\Enums\GenericStatusEnum;
use App\Exceptions\CommunityNameAlreadyTaken;
use App\Exceptions\InvalidAddCommunityAddSubscriberException;
use App\Exceptions\UploadFileException;
use App\Models\Community;
use App\Models\CommunityReport;
use App\Models\CommunitySetting;
use App\Models\CommunitySubscriber;
use App\Models\Post;
use App\Models\User;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait CommunityService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return Community::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    * Create community
    *
    * @param array $data
    * @return Community
    * @throws UploadFileException
    */
   public static function createCommunity(array $data): Community
   {
      DB::beginTransaction();

      try {
         /** @var User $user */
         $user = auth()->user();

         /** @var Community $community */
         $community = self::store([
            "designation" => $data["designation"],
            "description" => $data["description"],
            'uuid' => generate_uuid('communities'),
            'label_color' => $data['label_color'] ?? null,
            'visibility' => $data['visibility'],
            'create_by_user_id' => $user->{"id"},
         ]);

         if (isset($data["avatar"])) {
            self::handleUploadCommunityAvatar($community, $data, $community->{"id"}, $user->{'id'});
         }

         // Add Community Admin
         CommunitySubscriber::store([
            "is_admin" => GenericStatusEnum::enable->value,
            "user_id" => $user->{"id"},
            "community_id" => $community->{"id"},
         ]);

         if (isset($data["community_subscribers"]) && $data["community_subscribers"] != null) {
            foreach ($data["community_subscribers"] as $subscriber) {
               CommunitySubscriber::store([
                  "is_admin" => GenericStatusEnum::disable->value,
                  "user_id" => $subscriber,
                  "community_id" => $community->{"id"},
               ]);
            }
         }

         DB::commit();
         $result = $community->refresh();
      } catch (\Exception $exception) {
         DB::rollBack();
         log_debug($exception, 'CommunityService::createCommunity Unable to create community');

         throw  $exception;
      }

      return $result;
   }

   /**
    * Edit community
    *
    * @param array $data
    * @return Community
    * @throws UploadFileException
    * @throws CommunityNameAlreadyTaken
    */
   public function editCommunity(array $data): Community
   {
      try {
         /** @var User $user */
         $user = auth()->user();
         $community = $this;

         $labelColor = $data['label_color'] ?? null;
         $avatarFile = $data['avatar'] ?? null;
         $designation = $data['designation'] ?? null;

         if(filled($designation)) {
            // Find by designation
            $communityCheck = self::findByDesignation($designation);
            if($communityCheck !== null && $communityCheck->{'id'} !== $community->{'id'}) {
               throw new CommunityNameAlreadyTaken(
                  message: __('errors.community_name_already_exist')
               );
            }
         }

         $community->updateService([
            "designation" => $designation ?? $community->{'designation'},
            "description" => $data["description"] ?? $community->{'description'},
            'visibility' => $data['visibility'] ?? $community->{'visibility'},
            'avatar' => filled($labelColor) ? null : $community->{'avatar'},
            'label_color' => $avatarFile !== null ? null : ($labelColor ?? $community->{'label_color'}),
         ]);

         if ($avatarFile !== null) {
            // TODO Delete old community avatar after uploading new one
            self::handleUploadCommunityAvatar($community, $data, $this->{"id"}, $user->{"id"});
         }

         $result = $community->refresh();
      } catch (\Exception $exception) {
         log_debug($exception, 'CommunityService::editCommunity Unable to edit community');
         throw  $exception;
      }

      return $result;
   }

   /**
    * Delete community
    *
    * @return Community
    */
   public function deleteCommunity(): Community
   {
      $community = $this;

      DB::beginTransaction();

      try {
         // Delete community subscribers
         $community->communitySubscribers()->delete();

         // Delete community membership requests
         $community->communityMembershipRequests()->delete();

         // Delete community settings
         $community->communitySetting()->delete();

         // Delete community reports
         $community->communityReport()->delete();

         // Delete community suspensions
         $community->communitySuspension()->delete();

         //Delete community posts
         $community->communityPost()->delete();

         // TODO Delete community avatar in storage
         // Delete community
         $community->delete();

         DB::commit();
      } catch (\Exception $exception) {
         DB::rollBack();
         log_debug($exception, "CommunityService::deleteCommunity");
      }

      return $community;
   }

   /**
    * Handle upload community avatar
    *
    * @param Community $community
    * @param array $data
    * @param $communityId
    * @param $userId
    * @throws UploadFileException
    */
   public static function handleUploadCommunityAvatar(Community $community, array $data, $communityId, $userId): void
   {
      // Get user's current document path
      $userCurrentDocPath = sprintf(config('torryme.paths.docs'), User::$prefixDir . $userId);

      // Get community path
      $verificationDocPath = sprintf($userCurrentDocPath . '/%s', Community::$prefixDir . $communityId);

      // Store community avatar
      $path = upload_image(DocumentTypeEnum::communityAvatar->value, $verificationDocPath, $data["avatar"]);

      if (!filled($path)) {
         // TODO Delete community avatar in case of error
         throw new UploadFileException(__('errors.unable_to_upload_community_avatar'));
      }

      // Update and add avatar
      $community->updateService([
         'avatar' => $path,
      ]);
   }

   /**
    * Find by id
    *
    * @param int $id
    * @return Builder|Builder[]|Collection|Model|null
    */
   public static function findById(int $id): Model|Collection|Builder|array|null
   {
      return Community::query()->find($id);
   }

   /**
    * Add subscriber to community
    *
    * @param array $data
    * @return Builder|Model
    * @throws InvalidAddCommunityAddSubscriberException
    */
   public static function addSubscriberToCommunity(array $data): Model|Builder
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         // Check if action is performed by community admin
         $isCommunityAdmin = CommunitySubscriber::checkIsCommunityAdminOrAlreadySubscriber(GenericStatusEnum::enable->value, $user->{"id"}, $data["community_id"]);

         if ($isCommunityAdmin == null) {
            throw new InvalidAddCommunityAddSubscriberException(__("errors.invalid_add_community_subscriber"));
         }

         // Check if user to be added is already a subscriber
         $subscriberAlreadyExist = CommunitySubscriber::checkIsCommunityAdminOrAlreadySubscriber(GenericStatusEnum::disable->value, $data["user_id"], $data["community_id"]);

         if ($subscriberAlreadyExist != null) {
            $result = $subscriberAlreadyExist;
         } else {
            // Add subscriber to community
            $communitySubscriber = CommunitySubscriber::addSubscriber($data);

            $result = $communitySubscriber->refresh();
         }
      } catch (\Exception $exception) {
         log_debug($exception, 'CommunityService::addSubscriberToCommunity');
         throw $exception;
      }

      return $result;
   }

   /**
    * Get users communities
    *
    * @param int $userId
    * @param int|null $pageNumber
    * @return array
    */
   public static function getUserCommunities(int $userId, int $pageNumber = null): array
   {
      $queryBuilder =
         Community::query()
            ->whereHas('communitySubscribers', function (Builder $builder) use ($userId) {
               $builder->where('user_id', $userId);
            })
            ->latest();

      $paginator = $queryBuilder->paginate(
         perPage: config("torryme.constants.items_per_page"),
         page: $pageNumber
      );

      return self::buildCommunityPaginator($paginator);
   }

   /**
    * Get suggested communities
    *
    * @param int $userId
    * @param int|null $pageNumber
    * @return array
    */
   public static function communitySuggestions(int $userId, int $pageNumber = null): array
   {
      $queryBuilder =
         Community::query()
            ->where('create_by_user_id', '<>', $userId)
            ->whereHas('communitySubscribers', function (Builder $builder) use ($userId) {
               $builder->where('user_id', '!=', $userId);
            })
            ->latest();

      $paginator = $queryBuilder->paginate(
         perPage: config("torryme.constants.items_per_page"),
         page: $pageNumber
      );

      return self::buildCommunityPaginator($paginator);
   }

   /**
    * Get community posts for feed
    *
    * @param int|null $page
    * @return LengthAwarePaginator
    */
   public function postsFeed(int $page = null): LengthAwarePaginator
   {
      return Post::getCommunityPostFeeds($this->{'id'}, $page);
   }

   /**
    * Build community paginator
    *
    * @param LengthAwarePaginator $paginator
    * @return array
    */
   public static function buildCommunityPaginator(LengthAwarePaginator $paginator): array
   {
      /** @var User $user */
      $user = auth()->user();

      $communitiesResultArray = array();
      $communities = $paginator->items();

      foreach ($communities as $community) {

         foreach ($community->{"communitySubscribers"} as $communitySubscriber) {
           if($communitySubscriber->{"user_id"} == $user->{"id"}) {
              $community->{"last_user_community_suspension"} = $communitySubscriber->lastCommunitySuspension();
           }
         }

         $community = $community->buildCommonSerialization();
         $communitiesResultArray[] = $community;
      }

      return default_paginator_format(
         $paginator->lastPage(),
         $paginator->total(),
         $paginator->currentPage(),
         'communities',
         $communitiesResultArray,
      );
   }

   /**
    * Try to serialize usual attributes
    *
    * @return Community
    */
   public function buildCommonSerialization(): Community
   {
      $community = $this;

      $community = $community->load([
         'creator.userDetail',
         'creator.business',
         'communitySubscribers.user.userDetail',
         'communitySubscribers.user.business'
      ]);
      $community->{'last_user_community_settings'} = $community->computeLastAuthUserCommunitySetting();

      return $community;
   }

   /**
    * Edit or create settings
    *
    * @param array $data
    * @return Model|Builder
    */
   public function editOrUpdateSettings(array $data): Model|Builder
   {
      return CommunitySetting::createOrUpdateSetting($this, $data);
   }

   /**
    * Compute last auth user community setting for THIS community
    *
    * @return Builder|Model|null
    */
   public function computeLastAuthUserCommunitySetting(): Builder|Model|null
   {
      /** @var ?User $user */
      $user = auth()->user();
      $lastUserCommunitySettings = null;

      if($user !== null) {
         $lastUserCommunitySettings = CommunitySetting::findByCommunityAndUser(
            $this->{'id'},
            $user->{'id'},
         );
      }

      return $lastUserCommunitySettings;
   }

   /**
    * Add community report
    *
    * @param array $data
    * @return Builder|Model
    */
   public function report(array $data): Model|Builder
   {
      return CommunityReport::createReport($this, $data);
   }

   /** Find by designation
    *
    * @param string $designation
    * @return Builder|Model|null
    */
   public function findByDesignation(string $designation): Model|Builder|null
   {
      return
         Community::query()
            ->whereRaw(
               'LOWER(`designation`) LIKE ?',
               [trim(strtolower($designation)).'%']
            )
            ->first();
   }
}
