<?php

namespace App\Services;

use App\Enums\CommunityMembershipRequestStatusEnum;
use App\Enums\GenericStatusEnum;
use App\Exceptions\DoubleMembershipRequestException;
use App\Exceptions\InvalidCommunityMembershipCancellationAttempt;
use App\Exceptions\InvalidCommunityMembershipDeclinationAttempt;
use App\Exceptions\InvalidCommunityMembershipValidationAttempt;
use App\Exceptions\InvalidMembershipRequestAttempt;
use App\Models\CommunityMembershipRequest;
use App\Models\CommunitySubscriber;
use App\Models\User;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Mockery\Exception;

trait CommunityMembershipRequestService
{
   /**
    * Store new record
    *
    * @param array $data
    * @return Builder|Model
    */
   public static function store(array $data): Model|Builder
   {
      return CommunityMembershipRequest::query()->create($data);
   }

   /**
    * Update existing record
    *
    * @param array $data
    * @return mixed
    */
   public function updateService(array $data): mixed
   {
      return tap($this)->update($data);
   }

   /**
    * Check membership request owner
    *
    * @param int $userId
    * @throws InvalidCommunityMembershipCancellationAttempt
    */
   public function checkRequestOwner(int $userId): void
   {
      if ($this->{"user_id"} !== $userId) {
         throw new InvalidCommunityMembershipCancellationAttempt(__("errors.invalid_membership_cancellation_action"));
      }
   }

   /**
    * Add membership request
    *
    * @param array $data
    * @return Builder|Model
    * @throws DoubleMembershipRequestException
    */
   public static function addMembershipRequest(array $data): Model|Builder
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         // Check if membership request already exists
         $requestExists = self::checkRequestAlreadyExist($user->{"id"}, $data["community_id"]);
         if ($requestExists != null) {
            if($requestExists->{"status"} == CommunityMembershipRequestStatusEnum::pending->value) {
               throw  new DoubleMembershipRequestException(__("errors.double_membership_request_action"));
            }

            if($requestExists->{"status"} == CommunityMembershipRequestStatusEnum::approved->value) {
               throw  new DoubleMembershipRequestException(__("errors.already_community_member"));
            }
         }

         $membershipRequest = self::store([
            "status" => CommunityMembershipRequestStatusEnum::pending->value,
            "user_id" => $user->{"id"},
            "community_id" => $data["community_id"]
         ]);

         $results = $membershipRequest->refresh();
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityMembershipRequestService::addMembershipRequest");
         throw $exception;
      }

      return $results;
   }

   /**
    * Find by id
    *
    * @param int $id
    * @return Builder|Builder[]|Collection|Model|null
    */
   public static function findById(int $id): Model|Collection|Builder|array|null
   {
      return CommunityMembershipRequest::query()->find($id);
   }

   /**
    * Check is record already exist
    *
    * @param int $userId
    * @param int $communityId
    * @return Model|Builder|null
    */
   public static function checkRequestAlreadyExist(int $userId, int $communityId): Model|Builder|null
   {
      $query = CommunityMembershipRequest::query()->where("user_id", $userId)->where('community_id', $communityId)->first();

      if($query != null) {
         if($query->{"status"} == CommunityMembershipRequestStatusEnum::canceled->value || $query->{"status"} == CommunityMembershipRequestStatusEnum::declined->value) {
            $query->delete();
            return  null;
         }
      }

      return  $query;
   }

   /**
    * Cancel membership request
    *
    * @return CommunityMembershipRequest
    * @throws InvalidCommunityMembershipCancellationAttempt
    */
   public function cancelMembershipRequest(): CommunityMembershipRequest
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         // Check user cancelling request is owner of request
         self::checkRequestOwner($user->{'id'});

         $this->updateService([
            "status" => CommunityMembershipRequestStatusEnum::canceled->value,
         ]);

         $result = $this->refresh();
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityMembershipRequestService::cancelMembershipRequest");
         throw $exception;
      }

      return $result;
   }

   /**
    * Decline membership request
    *
    * @return CommunityMembershipRequest
    * @throws InvalidCommunityMembershipDeclinationAttempt
    */
   public function declineMembershipRequest(): CommunityMembershipRequest
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         // Check if action is performed by community admin
         $isCommunityAdmin = CommunitySubscriber::checkIsCommunityAdminOrAlreadySubscriber(GenericStatusEnum::enable->value, $user->{"id"}, $this->{"community_id"});

         if (!$isCommunityAdmin) {
            throw new InvalidCommunityMembershipDeclinationAttempt(__("errors.invalid_membership_declination_action"));
         }

         $this->updateService([
            "status" => CommunityMembershipRequestStatusEnum::declined->value,
         ]);

         $result = $this->refresh();
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityMembershipRequestService::declineMembershipRequest");
         throw $exception;
      }

      return $result;
   }

   /**
    * Approve membership request and add community subscriber
    *
    * @return Builder|Model
    * @throws InvalidCommunityMembershipValidationAttempt
    */
   public function approveMembershipRequest(): Model|Builder
   {
      try {
         $data = array();

         /** @var User $user */
         $user = auth()->user();

         // Check if action is performed by community admin
         $isCommunityAdmin = CommunitySubscriber::checkIsCommunityAdminOrAlreadySubscriber(GenericStatusEnum::enable->value, $user->{"id"}, $this->{"community_id"});

         if (!$isCommunityAdmin) {
            throw new InvalidCommunityMembershipValidationAttempt(__("errors.invalid_membership_declination_action"));
         }

         $this->updateService([
            "status" => CommunityMembershipRequestStatusEnum::approved->value,
         ]);

         // Add values to array data
         $data["user_id"] = $this->{"user_id"};
         $data["community_id"] = $this->{"community_id"};

         // Add community subscriber
         $communitySubscriber = CommunitySubscriber::addSubscriber($data);

         $result = $communitySubscriber;
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityMembershipRequestService::declineMembershipRequest");
         throw $exception;
      }

      return $result;
   }

   /**
    * Get membership requests
    *
    * @param int $communityId
    * @param int|null $pageNumber
    * @return array
    * @throws InvalidMembershipRequestAttempt
    */
   public static function memberRequests(int $communityId, int $pageNumber = null): array
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         // Check if action is performed by community admin
         $isCommunityAdmin = CommunitySubscriber::checkIsCommunityAdminOrAlreadySubscriber(GenericStatusEnum::enable->value, $user->{"id"}, $communityId);

         //Throw exception is user is not community admin
         if (!$isCommunityAdmin) {
            throw new InvalidMembershipRequestAttempt(__("errors.invalid_membership_request_action"));
         }

         $queryBuilder = CommunityMembershipRequest::query()->where('community_id', $communityId);

         $paginator = $queryBuilder->paginate(
            perPage: config("torryme.constants.items_per_page"),
            page: $pageNumber
         );

         $results = self::buildMembershipRequestPaginator($paginator);
      } catch (Exception $exception) {
         log_debug($exception, "CommunityService::memberRequests");
         throw $exception;
      }

      return $results;
   }

   /**
    * Get user membership requests
    *
    * @param int $userId
    * @param int|null $pageNumber
    * @return array
    */
   public static function getUserMembershipRequests(int $userId, int $pageNumber = null): array
   {
      $queryBuilder = CommunityMembershipRequest::query()->where('user_id', $userId);

      $paginator = $queryBuilder->paginate(
         perPage: config("torryme.constants.items_per_page"),
         page: $pageNumber
      );

      return self::buildMembershipRequestPaginator($paginator);
   }

   /**
    * Get user last membership request status
    *
    * @param int $communityId
    * @return Builder|Model|null
    */
   public static function getLastMembershipRequest(int $communityId): Builder|Model|null
   {
      /** @var ?User $user */
      $user = auth()->user();
      $membershipRequest = null;

      if($user !== null) {
         $membershipRequest =
            CommunityMembershipRequest::query()
               ->where('user_id', $user->{"id"})
               ->where('community_id', $communityId)
               ->first();

         $membershipRequest?->load('user.userDetail', 'user.business');

      }

      return $membershipRequest;
   }

   /**
    * Membership request paginator
    *
    * @param LengthAwarePaginator $paginator
    * @return array
    */
   public static function buildMembershipRequestPaginator(LengthAwarePaginator $paginator): array
   {
      $membershipRequestResultArray = array();
      $membershipRequests = $paginator->items();

      foreach ($membershipRequests as $membershipRequest) {
         $membershipRequest = $membershipRequest->load(['user.userDetail', 'user.business']);
         $membershipRequestResultArray[] = $membershipRequest;
      }

      return default_paginator_format(
         $paginator->lastPage(),
         $paginator->total(),
         $paginator->currentPage(),
         'membership_requests',
         $membershipRequestResultArray,
      );
   }
}
