<?php

namespace App\Services;

use App\Enums\UserTypeEnum;
use App\Exceptions\BusinessCreationException;
use App\Models\Business;
use App\Models\BusinessAffiliate;
use App\Models\City;
use App\Models\Country;
use App\Models\Product;
use App\Models\ProductRating;
use App\Models\User;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait BusinessService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Business::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Find by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return Business::query()->find($id);
    }

    /**
     * Cech if user already has=ve a business
     *
     * @param User $user
     * @return bool
     */
    public static function alreadyHasBusiness(User $user): bool
    {
        return Business::query()->where("owner_user_id", $user->{"id"})->exists();
    }

    /**
     * Create business
     *
     * @param array $data
     * @return User
     * @throws BusinessCreationException
     * @throws Exception
     */
    public static function createBusiness(array $data): User
    {
        /** @var User $user */
        $user = auth()->user();

        if (self::alreadyHasBusiness($user)) {
            throw new  BusinessCreationException(__("errors.business_already_exist_for_user"));
        }

        DB::beginTransaction();
        try {

            $businessCountryId = null;
            $businessCityId = null;

            if ($data["country_code"] != null) {

                if ($data["city_name"] != null) {
                    $searchCountry = Country::findByCountryCode($data["country_code"]);
                    if ($searchCountry != null) {
                        $searchCity = City::getByCountryIdAndCityName($searchCountry->{"id"}, $data["city_name"]);
                        if ($searchCity != null) {
                            $businessCityId = $searchCity->{"id"};
                            $businessCountryId = $searchCity->{"country_id"};
                        } else {
                            $createCity = City::store([
                                "designation" => $data["city_name"],
                                "code" => $data["city_name"],
                                "country_id" => $searchCountry->{"id"},
                            ]);
                            $businessCityId = $createCity->{"id"};
                            $businessCountryId = $createCity->{"country_id"};
                        }
                    } else {
                        //Get country object
                        $countryObject = Country::getCountryInfoByCode($data["country_code"]);

                        //Create new country
                        if ($countryObject != null) {
                            $createCountry = Country::store([
                                "designation" => $countryObject->{"designation"},
                                "dial_code" => $countryObject->{"dial_code"},
                                "country_code" => $countryObject->{"country_code"},
                            ]);

                            //Create new city
                            $createCityFromCountryObject = City::store([
                                "designation" => $data["city_name"],
                                "code" => $data["city_name"],
                                "country_id" => $createCountry->{"id"},
                            ]);

                            $businessCityId = $createCityFromCountryObject->{"id"};
                            $businessCountryId = $createCityFromCountryObject->{"country_id"};

                        }
                    }
                }
            }

            // Create business
            self::store([
                "designation" => $data["designation"],
                "geographic_location" => $data["geographic_location"],
                "address" => $data["address"],
                "owner_user_id" => $user->{"id"},
                "business_category_id" => $data["business_category_id"],
                "country_id" => $businessCountryId,
                "city_id" => $businessCityId,
            ]);

            // Update user type
            $user->updateService(["user_type" => UserTypeEnum::business->value]);

            DB::commit();
            $result = $user->refresh()->load(['userDetail', 'business']);
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'BusinessService::createBusiness');

            DB::rollBack();
            throw $exception;
        }

        return $result;
    }

    /**
     * Checks if the business is the goods type
     *
     * @return bool
     */
    public function isGoodsBusiness(): bool
    {
        return $this->{'businessCategory'}->{'code'} === config('torryme.business_category_codes.goods');
    }

    /**
     * Checks if the business is the service type
     *
     * @return bool
     */
    public function isServiceBusiness(): bool
    {
        return $this->{'businessCategory'}->{'code'} === config('torryme.business_category_codes.service');
    }

    /**
     * Checks if the business is the cinema type
     *
     * @return bool
     */
    public function isCinemaBusiness(): bool
    {
        return $this->{'businessCategory'}->{'code'} === config('torryme.business_category_codes.cinema');
    }

    /**
     * Checks if the business is the music type
     *
     * @return bool
     */
    public function isMusicBusiness(): bool
    {
        return $this->{'businessCategory'}->{'code'} === config('torryme.business_category_codes.music');
    }

    /**
     * Checks if the business is the library type
     *
     * @return bool
     */
    public function isLibraryBusiness(): bool
    {
        return $this->{'businessCategory'}->{'code'} === config('torryme.business_category_codes.library');
    }

    /**
     * Checks if the business is the game type
     *
     * @return bool
     */
    public function isGameBusiness(): bool
    {
        return $this->{'businessCategory'}->{'code'} === config('torryme.business_category_codes.game');
    }

    /**
     * Checks if the business is affiliate
     *
     * @return bool
     */
    public function isAffiliate(): bool
    {
        return $this->{'businessCategory'}->{'code'} === config('torryme.business_category_codes.affiliate');
    }

    /**
     * Get last affiliation
     *
     * @return mixed
     */
    public function lastAffiliation(): mixed
    {
        $result =
            BusinessAffiliate::query()
                ->where('business_affiliated_id', $this->{'id'})
                ->where('expired_at', '>', now())
                ->orderByDesc('created_at')
                ->first();

        return $result?->{'businessAffiliate'};
    }

    /**
     * Get business products review
     *
     * @param int|null $page
     * @return array
     */
    public function reviews(int $page = null): array
    {
        $business = $this;
        $products = Product::getProductsWithRatings($business, $page);

        $results = array();
        foreach ($products as $product) {
            $singleResult = [
                'product' => $product->computeDetailsForReviews(),
                'reviews' => ProductRating::allProductRatings($product, $page),
            ];

            $results[] = $singleResult;
        }

        return default_paginator_format(
            $products->lastPage(),
            $products->total(),
            $products->currentPage(),
            'reviews',
            $results
        );
    }

    /**
     * Total business reviews
     *
     * @return int
     */
    public function totalReviews(): int
    {
        $business = $this;

        return
            ProductRating::query()
            ->whereHas('product', function(Builder $builder) use($business) {
                $builder->where('business_id', $business->{'id'});
            })
            ->count();
    }

    /**
     * Get business by category
     *
     * @return Builder[]|Collection
     */
    public function getBusinessByCategory(): Collection|array
    {
        $businessCategoryCode = $this->{'businessCategory'}->{'code'};

        return Business::query()->whereNot("id", $this->{"id"})->whereHas("businessCategory",  function (Builder $b) use ($businessCategoryCode) {
            $b->where("code", $businessCategoryCode);
        })->limit(10)->get();
    }
}
