<?php

namespace App\Services;

use App\Enums\DocumentTypeEnum;
use App\Exceptions\InvalidBookPurchaseException;
use App\Models\Book;
use App\Models\Product;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\DB;

trait BookService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Book::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Check book is not hard copy
     *
     * @throws InvalidBookPurchaseException
     */
    public function isHardCpy(string $bookCode) {
        if($bookCode == config('torryme.business_sub_category_child_codes.hard_copy')) {
            throw  new InvalidBookPurchaseException(__("errors.invalid_book_purchase_attempt"));
        }
    }

    /**
     * Find by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return Book::query()->find($id);
    }

    /**
     * Try to add or update book cover
     *
     * @param UploadedFile $uploadedFile
     * @param string $basePath
     * @return Book
     */
    public function addBookCover(UploadedFile $uploadedFile, string $basePath): Book
    {
        // Try to upload ...
        $resultPath = upload_file_system(DocumentTypeEnum::bookCover->value, $basePath, $uploadedFile);

        if(filled($resultPath)) {
            $this->updateService(array(
                "cover" => $resultPath,
            ));
        }

        return $this->refresh();
    }

    /**
     * Try to add or update associated book file
     *
     * @param UploadedFile $uploadedFile
     * @param string $basePath
     * @return Book
     */
    public function addBookFile(UploadedFile $uploadedFile, string $basePath): Book
    {
        // Try to upload ...
        $resultPath = upload_file_system(DocumentTypeEnum::bookFile->value, $basePath, $uploadedFile);

        if(filled($resultPath)) {
            $this->updateService(array(
                "file_path" => $resultPath,
            ));
        }

        return $this->refresh();
    }

    /**
     * Create new book with cover file
     *
     * @param Product $product
     * @param string $basePath
     * @param UploadedFile $cover
     * @return Model|Builder|null
     */
    public static function createBookWithCover(Product $product, string $basePath, UploadedFile $cover): Model|Builder|null
    {
        $book = null;
        DB::beginTransaction();

        try {
            // Try to upload ...
            $resultPath = upload_file_system(DocumentTypeEnum::bookCover->value, $basePath, $cover);
            if(filled($resultPath)) {
                $book = Book::store(array(
                    "cover" => $resultPath,
                    'designation' => $product->{'designation'},
                    'description' => $product->{'description'},
                    'price_currency_id' => $product->{'price_currency_id'},
                    'price' => $product->{'price'},
                ));
            }

            if($book !== null) {
                $product->updateService([
                    "book_id" => $book->{'id'},
                ]);

                DB::commit();
                return $book->refresh();
            }

            DB::rollBack();
        } catch (\Exception $exception) {
            log_debug(exception: $exception,  prefix: 'BookService::createBookWithCover');

            DB::rollBack();
        }

        return $book;
    }

    /**
     * Create new book with file
     *
     * @param Product $product
     * @param string $basePath
     * @param UploadedFile $bookFile
     * @return Model|Builder|null
     */
    public static function createBookWithFile(Product $product, string $basePath, UploadedFile $bookFile): Model|Builder|null
    {
        $book = null;
        DB::beginTransaction();

        try {
            // Try to upload ...
            $resultPath = upload_file_system(DocumentTypeEnum::bookFile->value, $basePath, $bookFile);
            if(filled($resultPath)) {
                $book = Book::store(array(
                    "file_path" => $resultPath,
                    'designation' => $product->{'designation'},
                    'description' => $product->{'description'},
                    'price_currency_id' => $product->{'price_currency_id'},
                    'price' => $product->{'price'},
                ));
            }

            if($book !== null) {
                $product->updateService([
                    "book_id" => $book->{'id'},
                ]);

                DB::commit();
                return $book->refresh();
            }

            DB::rollBack();
        } catch (\Exception $exception) {
            log_debug(exception: $exception,  prefix: 'BookService::createBookWithFile');

            DB::rollBack();
        }

        return $book;
    }
}
