<?php

namespace App\Services;

use App\Exceptions\InvalidAccountBlockingAttempt;
use App\Models\BlockedAccounts;
use App\Models\User;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait BlockedAccountsService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return BlockedAccounts::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Block account
     *
     * @param int $blockedUserId
     * @return Builder|Model
     * @throws InvalidAccountBlockingAttempt
     */
    public static function blockAccount(int $blockedUserId): Model|Builder
    {
        try {
            /** @var User $user */
            $user = auth()->user();

            $blockedAlready = self::alreadyBlocked($user->{"id"}, $blockedUserId);

            if ($blockedAlready) {
                $blockedAccount = self::unblock($user->{"id"}, $blockedUserId);
            } else {
                if ($user->{"id"} == $blockedUserId) {
                    throw new InvalidAccountBlockingAttempt(__("errors.invalid_account_block_attempt"));
                }

                $blockedAccount = BlockedAccounts::store([
                    "blocked_user_id" => $blockedUserId,
                    "blocked_by_user_id" => $user->{"id"},
                ]);
            }

            $result = $blockedAccount;
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'BlockedAccountsService::blockAccount');
            throw $exception;
        }

        return $result;
    }

    /**
     * Unblock account
     *
     * @param int $unblockedUserId
     * @return null
     */
    public static function unblockAccount(int $unblockedUserId)
    {
        try {
            /** @var User $user */
            $user = auth()->user();

            $blockedAlready = self::alreadyBlocked($user->{"id"}, $unblockedUserId);
            if ($blockedAlready) {
                return self::unblock($user->{"id"}, $unblockedUserId);
            }
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'BlockedAccountsService::unblockAccount');
        }

        return null;
    }

    /**
     * Check if user has already been blocked
     *
     * @param int $userId
     * @param int $blockedUserId
     * @return bool
     */
    public static function alreadyBlocked(int $userId, int $blockedUserId): bool
    {
        return BlockedAccounts::query()
            ->where("blocked_by_user_id", $userId)
            ->where("blocked_user_id", $blockedUserId)->exists();
    }

    /**
     * Unblock account
     *
     * @param int $userId
     * @param int $blockedUserId
     * @return mixed
     */
    public static function unblock(int $userId, int $blockedUserId): mixed
    {
        return
            BlockedAccounts::query()
                ->where("blocked_by_user_id", $userId)
                ->where("blocked_user_id", $blockedUserId)
                ->delete();
    }

    /**
     * Get user's blocked accounts
     *
     * @param null $pageNumber
     * @return LengthAwarePaginator
     */
    public static function userBlockedAccounts($pageNumber = null): LengthAwarePaginator
    {
        /** @var User $user */
        $user = auth()->user();

        $queryBuilder =
            BlockedAccounts::query()
                ->with('blockedUser')
                ->where('blocked_by_user_id', $user->{"id"})
                ->orderByDesc("created_at");

      return  $queryBuilder->paginate(
            perPage: config("torryme.constants.items_per_page"),
            page: $pageNumber
        );
    }

    /**
     * Search block accounts
     *
     * @param string $query
     * @param int|null $pageNumber
     * @return LengthAwarePaginator
     */
    public static function searchBlockedAccounts(string $query, int $pageNumber = null): LengthAwarePaginator
    {
        /** @var User $user */
        $user = auth()->user();

        $queryBuilder = BlockedAccounts::query()
            ->where('blocked_by_user_id', $user->{"id"})
            ->whereHas('blockedUser', function (Builder $builder) use ($query) {
                $builder->where('user_name', 'LIKE', '%' . $query . '%')
                    ->orWhereHas('business', function (Builder $builder) use ($query) {
                        $builder->where('designation', 'LIKE', '%' . $query . '%');
                    })
                    ->orWhereHas('userDetail', function (Builder $builder) use ($query) {
                        $builder->where(
                            DB::raw("CONCAT(first_name, ' ', last_name)"), 'LIKE', "%" . $query . "%"
                        );
                    });
            });

        return $queryBuilder->paginate(
            perPage: config("torryme.constants.items_per_page"),
            page: $pageNumber
        );
    }

    /**
     * Serialize data for bocked account query search
     * @param $blockedAccounts
     * @return array
     */
    public static function computeForBlockedAccounts($blockedAccounts): array
    {
        $blockedAccountsArray = array();
        foreach ($blockedAccounts as $blockedAccount) {
            $blockedAccountsArray[] = $blockedAccount->{"blockedUser"}->load(['userDetail', 'business']);
        }

        return $blockedAccountsArray;
    }
}
