<?php

namespace App\Services;

use App\Enums\BusinessAffiliateStatusEnum;
use App\Exceptions\ShouldBeAffiliateMarketerException;
use App\Exceptions\ShouldNotBeAffiliateMarketerException;
use App\Models\AffiliateProgram;
use App\Models\Business;
use App\Models\BusinessAffiliate;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

trait AffiliateProgramService
{
    /**
     * Store new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return AffiliateProgram::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Find by id
     *
     * @param int $affiliateProgramId
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $affiliateProgramId): Model|Collection|Builder|array|null
    {
        return AffiliateProgram::query()->find($affiliateProgramId);
    }

    /**
     * Check request user is not an affiliate marketer
     *
     * @param Business $business
     * @throws ShouldNotBeAffiliateMarketerException
     */
    public static function shouldNotBeAffiliateMarketer(Business $business): void
    {
          if ($business->{"businessCategory"}->{'code'} === config("torryme.business_category_codes.affiliate")) {
            throw  new ShouldNotBeAffiliateMarketerException(__('errors.business_category_can_not_performed_action'));
        }
    }

    /**
     * Check request user is an affiliate marketer
     *
     * @param Business $affiliatedBusiness
     * @throws ShouldBeAffiliateMarketerException
     */
    public static function shouldBeAffiliateMarketer(Business $affiliatedBusiness): void
    {
        if ($affiliatedBusiness->{"businessCategory"}->{'code'} !== config("torryme.business_category_codes.affiliate")) {
            throw  new ShouldBeAffiliateMarketerException( __('errors.business_category_can_not_performed_action'));
        }
    }

    /**
     * Create affiliate program
     *
     * @param array $data
     * @param Business $business
     * @return Builder|Model
     * @throws ShouldNotBeAffiliateMarketerException
     */
    public static function createProgram(array $data, Business $business): Model|Builder
    {
        try {
            // Check request user is not an affiliate marketer
            self::shouldNotBeAffiliateMarketer($business);

            $affiliateProgram = AffiliateProgram::store([
                "designation" => $data["designation"],
                "description" => $data["description"],
                "percentage" => $data["percentage"],
                "started_at" => convert_datetime_to_utc($data["started_at"], $data[config('torryme.constants.time_zone_key')]),
                "active_days" => $data["active_days"],
                "business_id" => $business->{"id"},
            ]);

            $result = $affiliateProgram;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'AffiliateProgramService::createProgram');
            throw  $exception;
        }

        return $result;
    }

    /**
     * Get all valid affiliate programs
     *
     * @param Business $affiliatedBusiness
     * @param int|null $pageNumber
     * @return LengthAwarePaginator
     * @throws ShouldBeAffiliateMarketerException
     */
    public static function allValidPrograms(Business $affiliatedBusiness, int $pageNumber = null,): LengthAwarePaginator
    {
        try {
            // Check request user is an affiliate marketer
            self::shouldBeAffiliateMarketer($affiliatedBusiness);

            // Get programs user is already affiliated to
            $programsAffiliatedTo = BusinessAffiliate::query()->where('business_affiliated_id', $affiliatedBusiness->{"id"})->select('affiliate_program_id')->get();
            $programsAffiliatedIds = $programsAffiliatedTo->toArray();

            // Put affiliate program ids in array below
            $programsAffiliatedToIdArray = array();
            foreach ($programsAffiliatedIds as $programsAffiliatedId) {
                foreach ($programsAffiliatedId as $key => $value) {
                    $programsAffiliatedToIdArray[] = $value;
                }
            }

            // Get all programs except those in which user is already affiliated to
            $queryBuilder = AffiliateProgram::query();
            $queryBuilder = $queryBuilder->where(function (Builder $builder) use ($programsAffiliatedToIdArray) {
                $builder->where("status", BusinessAffiliateStatusEnum::active->value);
                $builder->whereNotIn('id', $programsAffiliatedToIdArray);
            })->orderByDesc('created_at');

            $paginator = $queryBuilder->paginate(
                perPage: config("torryme.constants.items_per_page"),
                columns: ['*'],
                pageName: 'page',
                page: $pageNumber
            );

            $result = $paginator;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'AffiliateProgramService::allValidPrograms');
            throw  $exception;
        }

        return $result;
    }

    /**
     * Get business programs with associated affiliates
     *
     * @param Business $business
     * @param int|null $pageNumber
     * @return LengthAwarePaginator
     * @throws ShouldNotBeAffiliateMarketerException
     */
    public static function businessProgramsWithAffiliates(Business $business, int $pageNumber = null,): LengthAwarePaginator
    {
        try {
            // Check request user is not an affiliate marketer
            self::shouldNotBeAffiliateMarketer($business);

            $queryBuilder = AffiliateProgram::query();
            $businessId = $business->{"id"};

            $queryBuilder = $queryBuilder->where(function (Builder $builder) use ($businessId) {
                $builder->where("business_id", $businessId);
            })->orderByDesc('created_at');

            $paginator = $queryBuilder->paginate(
                perPage: config("torryme.constants.items_per_page"),
                columns: ['*'],
                pageName: 'page',
                page: $pageNumber
            );

            $result = $paginator;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'AffiliateProgramService::businessProgramsWithAffiliates');
            throw  $exception;
        }

        return $result;
    }
}
