<?php

namespace App\RequestRules\Api;


use App\Enums\GenericStatusEnum;

class UserRules
{
    /**
     * Authentication rules
     *
     * @return array
     */
    public static function authenticationRules(): array
    {
        return [
            'telephone' => 'required|string',
            'password' => 'required|string',

            'native_device_id' => 'required|string|filled',
            'device_name' => 'required|string|filled',
            'model' => 'required|string|filled',
            'brand' => 'required|string|filled',
            'operating_system' => 'required|string|filled',
            'mac_address' => 'nullable',
            'ip_address' => 'nullable',
        ];
    }

    /**
     * Registration rules
     *
     * @return array
     */
    public static function registrationRules(): array
    {
        return [
            'first_name' => 'required|string|filled',
            'last_name' => 'required|string|filled',
            'date_of_birth' => ['required', 'date', 'date_format:' . config('torryme.constants.default_date_format')],
            'profile_picture' => ['nullable', 'image', 'max:' . config('torryme.constants.max_file_size')],
            'password' => 'required|string|filled',

            // Other ...
            'language_code' => 'required|string|filled',
            'country_code' => 'nullable|string',

            // ID
            'email' => 'nullable|email|unique:users,email',
            'telephone' => 'nullable|phone:AUTO,unique:users,telephone',
            'user_name' => 'required|string|filled|unique:users,user_name',

            // Device info
            'native_device_id' => 'required|string|filled',
            'device_name' => 'required|string|filled',
            'model' => 'required|string|filled',
            'brand' => 'required|string|filled',
            'operating_system' => 'required|string|filled',
            'mac_address' => 'nullable',
            'ip_address' => 'nullable',
        ];
    }

    /**Account configuration rues
     *
     * @return \string[][]
     */
    public static function accountConfig(): array
    {
        return [
            'currency_code' => 'sometimes|string|exists:currencies,code',
            'country_code' => 'sometimes|string|exists:countries,country_code',
            'language_code' => ['sometimes', 'in:' . implode(",", array_values(config('torryme.locales_keys'))),],
        ];
    }

    /**
     * Edit profile rules
     *
     * @return array
     */
    public static function editProfile(): array
    {
        return [
            'first_name' => 'sometimes|string',
            'last_name' => 'sometimes|string',
            'date_of_birth' => ['sometimes', 'date', 'date_format:' . config('torryme.constants.default_date_format')],
            'avatar' => ['sometimes', 'image', 'max:' . config('torryme.constants.max_file_size')],
            'business_designation' => 'sometimes|string',
            'description ' => 'sometimes|string',
            'user_name' => 'sometimes|string',
            'bio' => 'sometimes|string',
        ];
    }

    /**
     * Subscribe to another user rule
     *
     * @return string[]
     */
    public static function followUser(): array
    {
        return [
            'following_user_id' => 'required|numeric|exists:users,id',
        ];
    }

    /**
     * Delete follower
     *
     * @return string[]
     */
    public static function deleteFollower(): array
    {
        return [
            'follower_user_id' => 'required|numeric|exists:users,id',
        ];
    }

    /**
     * Change user primary phone number
     *
     * @return string[]
     */
    public static function changePhoneNumber(): array
    {
        return [
            'telephone' => 'required|phone:AUTO',
            'password' => 'required|string',
        ];
    }

    /**
     * Change user primary email
     *
     * @return string[]
     */
    public static function changeEmail(): array
    {
        return [
            'email' => 'required|email',
            'password' => 'required|string',
        ];
    }

    /**
     * Change password rules
     *
     * @return string[]
     */
    public static function changePassword(): array
    {
        return [
            'old_password' => 'required|string',
            'new_password' => 'required|string',
            'confirm_password' => 'required|string',
        ];
    }

    /**
     * Delete account rules
     *
     * @return string[]
     */
    public static function deleteAccount(): array
    {
        return [
            'reason_for_deleting' => 'nullable|string',
            'export_data' => ['nullable', 'in:' . implode(",", array(GenericStatusEnum::enable->value, GenericStatusEnum::disable->value))],
            'password' => 'required|string',
        ];
    }

    /**
     * Block account rule
     *
     * @return string[]
     */
    public static function blockAccount(): array
    {
        return [
            'blocked_user_id' => 'required|numeric|exists:users,id',
        ];
    }

    /**
     * Unblock account rule
     *
     * @return string[]
     */
    public static function unblockAccount(): array
    {
        return [
            'unblocked_user_id' => 'required|numeric|exists:users,id',
        ];
    }

}
