<?php

namespace App\RequestRules\Api;


use App\Enums\OrderStatusEnum;

class OrderRequestRules
{
   /**
    * Add rule
    *
    * @return string[]
    */
   public static function add(): array
   {
      return [
         "cart_ids" => "required|array|exists:cart_items,id",
         "coupon_code" => "nullable|exists:coupons,code",
      ];
   }

   /**
    * order list rule
    *
    * @return string[][]
    */
   public static function orderList(): array
   {
      return [
         'request_type' => [
            'required',
            'in:' . implode(",", array(
               OrderStatusEnum::all->value,
               OrderStatusEnum::incoming->value,
               OrderStatusEnum::outgoing->value,
               OrderStatusEnum::pending->value,
               OrderStatusEnum::active->value,
               OrderStatusEnum::completed->value,
               OrderStatusEnum::cancelled->value,
               OrderStatusEnum::requestShipping->value,
            ))
         ],
      ];
   }

   /**
    * Validate order rules
    *
    * @return array
    */
   public static function validate(): array
   {
      return [
         "order_id" => "required|exists:orders,id",
         "supplier_shipping_amount" => 'sometimes|numeric|gte:0',
         'delivery_date' => ['required', 'date', 'date_format:' . config('torryme.constants.default_date_time_format')],
      ];
   }

   /**
    * Pay order rules
    *
    * @return array
    */
   public static function pay(): array
   {
      return [
         "order_id" => "required|exists:orders,id",
         'wallet_pin' => ['required', 'string', 'min:' . config('torryme.constants.wallet_pin_length'), 'max:' . config('torryme.constants.wallet_pin_length')],
      ];
   }

   /**
    * Confirm delivery rule
    *
    * @return string[]
    */
   public static function confirmDelivery(): array
   {
      return [
         "order_id" => "required|exists:orders,id",
      ];
   }

   /**
    * Cancel order rules
    *
    * @return string[]
    */
   public static function cancelOrder(): array
   {
      return [
         "order_id" => "required|exists:orders,id",
         "reason_for_cancellation" => "nullable|string",
      ];
   }

   /**
    * Buy movie rules
    *
    * @return array
    */
   public static function movie(): array
   {
      return [
         "movie_id" => "required|exists:movies,id",
         'wallet_pin' => ['required', 'string', 'min:' . config('torryme.constants.wallet_pin_length'), 'max:' . config('torryme.constants.wallet_pin_length')],
         "coupon_code" => "nullable|exists:coupons,code",
      ];
   }

   /**
    * Buy series rules
    *
    * @return array
    */
   public static function series(): array
   {
      return [
         "series_id" => "required|exists:series,id",
         'wallet_pin' => ['required', 'string', 'min:' . config('torryme.constants.wallet_pin_length'), 'max:' . config('torryme.constants.wallet_pin_length')],
         "coupon_code" => "nullable|exists:coupons,code",
      ];
   }

   /**
    * Buy music rules
    *
    * @return array
    */
   public static function music(): array
   {
      return [
         "music_id" => "required|exists:music,id",
         'wallet_pin' => ['required', 'string', 'min:' . config('torryme.constants.wallet_pin_length'), 'max:' . config('torryme.constants.wallet_pin_length')],
         "coupon_code" => "nullable|exists:coupons,code",
      ];
   }

   /**
    * Buy music album rules
    *
    * @return array
    */
   public static function musicAlbum()
   {
      return [
         "music_album_id" => "required|exists:music_albums,id",
         'wallet_pin' => ['required', 'string', 'min:' . config('torryme.constants.wallet_pin_length'), 'max:' . config('torryme.constants.wallet_pin_length')],
         "coupon_code" => "nullable|exists:coupons,code",
      ];
   }

   /**
    * Buy book rules
    *
    * @return array
    */
   public static function books(): array
   {
      return [
         "book_id" => "required|exists:books,id",
         'wallet_pin' => ['required', 'string', 'min:' . config('torryme.constants.wallet_pin_length'), 'max:' . config('torryme.constants.wallet_pin_length')],
         "coupon_code" => "nullable|exists:coupons,code",
      ];
   }

   /**
    * Buy game rule
    *
    * @return array
    */
   public static function game()
   {
      return [
         "game_id" => "required|exists:games,id",
         'wallet_pin' => ['required', 'string', 'min:' . config('torryme.constants.wallet_pin_length'), 'max:' . config('torryme.constants.wallet_pin_length')],
         "coupon_code" => "nullable|exists:coupons,code",
      ];
   }

   /**
    * Compute coupon amount rules
    *
    * @return string[]
    */
   public static function computeCouponAmount(): array
   {
      return [
         "coupon_code" => "required|exists:coupons,code",
         "cart_ids" => "nullable|array|exists:cart_items,id",
         "movie_id" => "nullable|exists:movies,id",
         "series_id" => "nullable|exists:series,id",
         "music_id" => "nullable|exists:music,id",
         "music_album_id" => "nullable|exists:music_albums,id",
         "book_id" => "nullable|exists:books,id",
         "game_id" => "nullable|exists:games,id",
      ];
   }

}
