<?php

namespace App\Models;

use App\Http\Controllers\Api\Platform\StorageFileController;
use App\Services\SeriesSeasonService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SeriesSeason extends Model
{
    use HasFactory, SoftDeletes;
    use SeriesSeasonService;

    static string $prefixDir = "season_";

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'series_seasons';

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'designation',
        'description',
        'occurrence',
        'downloadable_price',
        'downloadable_price_per_episode',

        'series_id',
        'price_currency_id',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'series_id',
        'price_currency_id',
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = ['created_at', 'updated_at', 'deleted_at'];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['last_episode_occurrence', 'cover_url'];

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot(): void
    {
        parent::boot();
    }

    /**  Append attributes */

    public function getLastEpisodeOccurrenceAttribute(): ?string
    {
        $episode = $this->seasonEpisodes()->orderByDesc('occurrence')->first();

        return $episode !== null ? $episode->{'occurrence'} : 1;
    }

    public function getCoverUrlAttribute(): ?string
    {
        $fileName = sprintf(
            "%s_%s%s_thumbnail%s",
            Series::$prefixDir . $this->{'series_id'},
            self::$prefixDir,
            $this->{'occurrence'},
            config('torryme.constants.default_image_extension')
        );

        return route('serve.doc', [
            'resourceId' => $this->{'id'},
            'q' => StorageFileController::seriesSeasonCover,
            'file' => $fileName
        ]);
    }

    /** Attributes used outside the class to create modify the resource */

    /**
     * Get all episode file size
     *
     * @return int
     */
    public function totalSize(): int
    {
        $totalSize = 0;
        $episodes = $this->{'seasonEpisodes'};

        foreach ($episodes as $episode) {
            $totalSize = $totalSize + $episode->episodeSize();
        }

        return $totalSize;
    }

    /** Relations */

    public function seasonEpisodes(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(SeasonEpisode::class, 'series_season_id');
    }

    public function series(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Series::class, 'series_id');
    }

    public function currency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Currency::class, 'price_currency_id');
    }
}
