<?php

namespace App\Models;

use App\Http\Controllers\Api\Platform\StorageFileController;
use App\Services\SeasonEpisodeService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;

class SeasonEpisode extends Model
{
    use HasFactory, SoftDeletes;
    use SeasonEpisodeService;

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'season_episodes';

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'designation',
        'description',
        'occurrence',
        'file_path',

        'series_season_id'
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'series_season_id'
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = ['created_at', 'updated_at', 'deleted_at'];

    /**
     * The accessors to append to the model's array form.
     *
     * @var array
     */
    protected $appends = ['episode_url'];

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot(): void
    {
        parent::boot();
    }

    /**  Append attributes */

    public function getEpisodeUrlAttribute(): ?string
    {
        if (filled($this->{'file_path'})) {
            return route('serve.doc', [
                'resourceId' => $this->{'id'},
                'q' => StorageFileController::episodeFile,
                'file' => $this->{'file_path'}
            ]);
        }

        return null;
    }

    /** Attributes used outside the class to create modify the resource */

    /**
     * Get episode duration
     *
     * @return int
     */
    public function episodeDuration(): int
    {
        if(filled($this->{'file_path'})) {
            return file_duration(
                $this->buildEpisodeFilePath($this->{'file_path'}),
            );
        }

        return 0;
    }

    /**
     * Get episode file size
     *
     * @return int
     */
    public function episodeSize(): int
    {
        $storageOption = config('torryme.storage_option');

        if(filled($this->{'file_path'})) {
            $filePathDir = $this->buildEpisodeFilePath($this->{'file_path'});

            if(Storage::disk($storageOption)->exists($filePathDir)) {
                return File::size(storage_path('app/' . $filePathDir));
            }
        }

        return 0;
    }

    /**
     * Build episode file path
     *
     * @param string $path
     * @return string|null
     */
    public function buildEpisodeFilePath(string $path): ?string
    {
        $product = $this->{'seriesSeason'}->{'series'}->{'product'};
        $ownerUserId = $product->{'business'}->{'owner_user_id'};
        $businessId = $product->{'business_id'};
        $productId = $product->{'id'};
        $seriesSeason = $this->{'seriesSeason'};

        return sprintf(
            config('torryme.paths.docs'),
            sprintf(
                '%s%s/%s%s/%s%s/%s%s/%s',
                User::$prefixDir, $ownerUserId,
                Business::$prefixDir, $businessId,
                Product::$prefixDir, $productId,
                SeriesSeason::$prefixDir, $seriesSeason->{'occurrence'},
                $path
            )
        );
    }

    /** Relations */

    public function seriesSeason(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(SeriesSeason::class, 'series_season_id');
    }
}
