<?php

namespace App\Models;

use App\Services\ProductService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Product extends Model
{
    use HasFactory, SoftDeletes;
    use ProductService;

    static string $prefixDir = "product_";

    /**
     * The database table used by the model.
     *
     * @var string
     */
    protected $table = 'products';

    /**
     * Attributes that should be mass-assignable.
     *
     * @var array
     */
    protected $fillable = [
        'designation',
        'description',
        'quantity',
        'price',
        'min_order',
        'max_order',
        'delivery_in_days',
        'activated_at',
        'published_at',
        'creation_status',
        'unit',

        'business_id',
        'price_currency_id',
        'business_category_id',
        'business_sub_category_id',
        'business_sub_category_child_id',
        'business_sub_category_grand_child_id',
        'series_id',
        'movie_id',
        'book_id',
        'music_album_id',
        'music_id',
        'game_id',
    ];

    /**
     * The attributes excluded from the model's JSON form.
     *
     * @var array
     */
    protected $hidden = [
        'business_id',
        'business_category_id',
        'business_sub_category_id',
        'business_sub_category_child_id',
        'business_sub_category_grand_child_id',
        'series_id',
        'movie_id',
        'book_id',
        'music_album_id',
        'music_id',
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'activated_at' => 'datetime',
        'published_at' => 'datetime',
        'created_at' => 'datetime',
    ];

    /**
     * The attributes that should be mutated to dates.
     *
     * @var array
     */
    protected $dates = ['created_at', 'updated_at', 'deleted_at', 'activated_at', 'published_at'];

    /**
     * The "booting" method of the model.
     *
     * @return void
     */
    public static function boot(): void
    {
        parent::boot();
    }

    /** Attributes used outside the class to create modify the resource */

    /**
     * Build attachment directory files path
     *
     * @param string $path
     * @return string
     */
    public function buildAttachmentDirPath(string $path): string
    {
        $business = $this->{'business'};
        $ownerUserId = $business->{'owner_user_id'};

        return sprintf(
            config('torryme.paths.docs'),
            sprintf(
                '%s_%s/%s_%s/%s_%s/%s',
                User::$prefixDir, $ownerUserId,
                Business::$prefixDir, $this->{'business_id'},
                Product::$prefixDir, $this->{'id'},
                $path,
            )
        );
    }

    /** Relations */

    public function businessCategory(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(BusinessCategory::class, 'business_category_id');
    }

    public function businessSubCategory(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(BusinessSubCategory::class, 'business_sub_category_id');
    }

    public function businessSubCategoryChild(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(BusinessSubCategoryChild::class, 'business_sub_category_child_id');
    }

    public function businessSubCategoryGrandChild(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(BusinessSubCategoryChild::class, 'business_sub_category_grand_child_id');
    }

    public function book(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Book::class);
    }

    public function movie(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Movie::class);
    }

    public function series(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Series::class);
    }

    public function musicAlbum(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(MusicAlbum::class);
    }

    public function business(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Business::class, "business_id");
    }

    public function currency(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Currency::class, "price_currency_id");
    }

    public function music(): \Illuminate\Database\Eloquent\Relations\BelongsTo
    {
        return $this->belongsTo(Music::class, "music_id");
    }

    public function tags(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(ProductHashTag::class);
    }

    public function productAttachments(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(ProductAttachment::class);
    }

    public function productAttributes(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(ProductAttribute::class);
    }

    public function userFollowerFollowingServices(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(Business::class, 'id');
    }

    public function productRating(): \Illuminate\Database\Eloquent\Relations\HasMany
    {
        return $this->hasMany(ProductRating::class);
    }

}
