<?php

namespace App\Models;

use App\Http\Controllers\Api\Platform\StorageFileController;
use App\Services\PostAttachmentService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class PostAttachment extends Model
{
   use HasFactory, SoftDeletes;
   use PostAttachmentService;

   /**
    * The database table used by the model.
    *
    * @var string
    */
   protected $table = 'post_attachments';

   /**
    * Attributes that should be mass-assignable.
    *
    * @var array
    */
   protected $fillable = [
      'path',
      'type',

      'post_id',
   ];

   /**
    * The attributes excluded from the model's JSON form.
    *
    * @var array
    */
   protected $hidden = [
      'post_id',
   ];

   /**
    * The attributes that should be casted to native types.
    *
    * @var array
    */
   protected $casts = [];

   /**
    * The attributes that should be mutated to dates.
    *
    * @var array
    */
   protected $dates = ['created_at', 'updated_at', 'deleted_at'];

   /**
    * The accessors to append to the model's array form.
    *
    * @var array
    */
   protected $appends = ['attachment_url'];

   /**
    * The "booting" method of the model.
    *
    * @return void
    */
   public static function boot(): void
   {
      parent::boot();
   }

   /** Attributes used outside the class to create modify the resource */

   /**
    * Compute attachment url
    *
    * @return string|null
    */
   public function getAttachmentUrlAttribute(): ?string
   {
      if (filled($this->{'path'})) {
         return route('serve.doc', [
            'resourceId' => $this->{'id'},
            'q' => StorageFileController::postAttachment,
            'file' => $this->{'path'}
         ]);
      }

      return null;
   }

   /**
    * Build attachment directory file path
    *
    * @param string|null $path
    * @return string|null
    */
   public function buildAttachmentFilesPath(string $path = null): ?string
   {
      $post = $this->{'post'};
      $user = $post->{'user'} ?? $post->{'business'}->{'ownerUser'};

      if($post->{'community_id'} !== null) {
         return sprintf(
            config('torryme.paths.docs'),
            sprintf(
               '%s%s/%s%s/%s%s/%s',
               User::$prefixDir, $user->{'id'},
               Community::$prefixDir, $post->{'community_id'},
               Post::$prefixDir, $post->{'id'},
               $path ?? $this->{'path'}
            )
         );
      }

      return sprintf(
         config('torryme.paths.docs'),
         sprintf(
            '%s%s/%s%s/%s',
            User::$prefixDir, $user->{'id'},
            Post::$prefixDir, $post->{'id'},
            $path ?? $this->{'path'}
         )
      );
   }

   /** Relations */

   public function post(): \Illuminate\Database\Eloquent\Relations\BelongsTo
   {
      return $this->belongsTo(Post::class);
   }
}
