<?php

namespace App\Models;

use App\Concerns\TranslatorConcern;
use App\Services\PostService;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Post extends Model
{
   use HasFactory, SoftDeletes;
   use PostService;
   use TranslatorConcern;

   static string $prefixDir = "posts/post_";

   /**
    * The database table used by the model.
    *
    * @var string
    */
   protected $table = 'posts';

   /**
    * Attributes that should be mass-assignable.
    *
    * @var array
    */
   protected $fillable = [
      'content',
      'turn_on_comment',
      'comments',
      'likes',
      'shares',
      'saves',
      'views',
      'completed_at',
      'timeline_type',

      'community_id',
      'product_id',
      'business_id',
      'user_id',
      'reposted_post_id',
   ];

   /**
    * The attributes excluded from the model's JSON form.
    *
    * @var array
    */
   protected $hidden = [
      'product_id',
      'business_id',
      'user_id',
      'reposted_post_id',
      'community_id',
   ];

   /**
    * The attributes that should be casted to native types.
    *
    * @var array
    */
   protected $casts = [];

   /**
    * The attributes that should be mutated to dates.
    *
    * @var array
    */
   protected $dates = ['created_at', 'updated_at', 'deleted_at'];

   /**
    * The accessors to append to the model's array form.
    *
    * @var array
    */
   protected $appends = ['shared_post_url'];

   /**
    * @return string
    */
   public function getSharedPostUrlAttribute(): string
   {
      return $this->buildSharedPostUrl();
   }

   /**
    * Build shared post url
    *
    * @return string
    */
   public function buildSharedPostUrl(): string
   {
      return build_share_post_url($this->{"id"});
   }

   /**
    * The "booting" method of the model.
    *
    * @return void
    */
   public static function boot(): void
   {
      parent::boot();
   }

   /** Attributes used outside the class to create modify the resource */

   /**
    * Full path to post assets
    *
    * @return string
    */
   public function buildAttachmentDir(): string
   {
      if($this->{'community'} !== null) {
         return $this->buildAttachmentDirForCommunity();
      }

      $userId = $this->{'user_id'} ?? $this->{'business'}->{'owner_user_id'};

      return sprintf(
         User::$prefixDir . "%s/" . self::$prefixDir . "%s",
         $userId,
         $this->{'id'},
      );
   }

   /**
    * Full path to community post assets
    *
    * @return string
    */
   public function buildAttachmentDirForCommunity(): string
   {
      $userId = $this->{'user_id'} ?? $this->{'business'}->{'owner_user_id'};

      return sprintf(
         User::$prefixDir . "%s/" . Community::$prefixDir . "%s/" . self::$prefixDir . "%s",
         $userId,
         $this->{'community_id'},
         $this->{'id'},
      );
   }

   /** Other methods */

   /**
    * Get post owner
    * @return mixed
    */
   public function owner(): mixed
   {
      return $this->{"business"} != null ? $this->{"business"}->{"ownerUser"} : $this->{"user"};
   }

   /** Relations */

   public function user(): \Illuminate\Database\Eloquent\Relations\BelongsTo
   {
      return $this->belongsTo(User::class);
   }

   public function business(): \Illuminate\Database\Eloquent\Relations\BelongsTo
   {
      return $this->belongsTo(Business::class);
   }

   public function product(): \Illuminate\Database\Eloquent\Relations\BelongsTo
   {
      return $this->belongsTo(Product::class);
   }

   public function repostedPost(): \Illuminate\Database\Eloquent\Relations\BelongsTo
   {
      return $this->belongsTo(Post::class, 'reposted_post_id')->withTrashed();
   }

   public function postAttachments(): \Illuminate\Database\Eloquent\Relations\HasMany
   {
      return $this->hasMany(PostAttachment::class);
   }

   public function postHags(): \Illuminate\Database\Eloquent\Relations\HasMany
   {
      return $this->hasMany(PostHashTag::class);
   }

   public function postMentions(): \Illuminate\Database\Eloquent\Relations\HasMany
   {
      return $this->hasMany(PostMention::class);
   }

   public function community(): \Illuminate\Database\Eloquent\Relations\BelongsTo
   {
      return $this->belongsTo(Community::class);
   }
}
