<?php

namespace App\Http\Controllers\Api\Platform;

use App\Enums\TransactionTypesNum;
use App\Exceptions\InsufficientWalletBalanceException;
use App\Exceptions\InvalidPasswordException;
use App\Exceptions\WalletBlockedException;
use App\Exceptions\WalletCurrencyChangeException;
use App\Exceptions\WalletPinAlreadyUsedException;
use App\Exceptions\WalletRemainingPinAttemptsException;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Wallet;
use App\RequestRules\Api\WalletRules;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class WalletController extends Controller
{
   /**
    * Get auth user wallet
    *
    * @param Request $request
    * @return Response|Application|ResponseFactory
    */
   public function walletStatus(Request $request): Response|Application|ResponseFactory
   {
      /** @var User $user */
      $user = auth()->user();

      /** @var Wallet $authWallet */
      $authWallet = $user->{"wallet"}->load('currency');

      $result = array("wallet" => $authWallet,);

      if ($request->has('with-transactions') && $request->get('with-transactions') == 1) {
         $result['transactions'] = $authWallet->transactions(pageNumber: 1);
      }

      return api_response(100, "Okay", $result);
   }

   /**
    * Create wallet pin
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function setPin(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(WalletRules::setPin());

      /** @var User $user */
      $user = auth()->user();

      try {
         // Get user wallet
         /** @var Wallet $wallet */
         $wallet = $user->{"wallet"};

         // Check if wallet pin is null
         if ($wallet->pinNotSet()) {
            $result = $wallet->setPin($request->get('wallet_pin'));

            if ($result == null) {
               return api_response(120, __("errors.unknown_error"));
            }

            return api_response(100, "Okay", $result->load('currency'));
         }
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletController::setPin');
         return api_response(120, __("errors.unknown_error"));
      }

      return api_response(120, __("errors.wallet_pin_already_set"));
   }

   /**
    * Update wallet pin of current user
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function updatePin(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(WalletRules::updatePin());

      $data = $request->all();

      /** @var User $user */
      $user = auth()->user();

      try {
         // Get user wallet
         /** @var Wallet $wallet */
         $wallet = $user->{"wallet"};

         $result = $wallet->updatePin($data);

         if ($result == null) {
            return api_response(120, __("errors.unknown_error"));
         }

         return api_response(100, "Okay", $result->load('currency'));

      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletController::updatePin');

         if ($exception instanceof WalletPinAlreadyUsedException) {
            return api_response(120, $exception->getMessage());
         }

         $walletException = self::returnWalletException($exception);
         if ($walletException != null) {
            return $walletException;
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Check wallet pin
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function checkPin(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(WalletRules::checkPin());

      /** @var User $user */
      $user = auth()->user();

      try {
         // Get user wallet
         /** @var Wallet $wallet */
         $wallet = $user->{"wallet"};

         //Check current pin match
         $wallet->verifyWalletPin($request->get("wallet_pin"));

         return api_response(100, "Okay");

      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletController::checkPin');

         $walletException = self::returnWalletException($exception);
         if ($walletException != null) {
            return $walletException;
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Find wallet by wallet number
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function findByWalletNumber(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(WalletRules::findByWalletNumber());

      /** @var User $user */
      $user = auth()->user();

      //Get wallet number of current auth user
      $authUserWalletNumber = $user->{"wallet"}->{"wallet_number"};

      $result = Wallet::findByWalletNumber($request->get("wallet_number"), $authUserWalletNumber);

      if ($result == null) {
         return api_response(120, __("errors.invalid_transfer_operation"));
      }

      $user = $result->{"user"};

      return api_response(100, "Okay", $user->load('userDetail'));
   }

   /**
    * Transfer funds to account
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function transferToAccount(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(WalletRules::transferToAccount());

      $data = $request->all();

      /** @var User $user */
      $user = auth()->user();

      try {
         //Get user wallet
         /** @var Wallet $wallet */
         $wallet = $user->{"wallet"};

         $result = $wallet->transferToAccount($data);

         return api_response(100, "Okay", $result);

      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletController::transferToAccount');

         if ($exception instanceof InsufficientWalletBalanceException) {
            return api_response(120, $exception->getMessage());
         }

         $walletException = self::returnWalletException($exception);
         if ($walletException != null) {
            return $walletException;
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Deposit to account
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function deposit(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(WalletRules::deposit());

      $data = $request->all();

      /** @var User $user */
      $user = auth()->user();

      try {
         // Get user wallet
         /** @var Wallet $wallet */
         $wallet = $user->{"wallet"};

         $result = $wallet->accountDeposit($data);

         return api_response(100, "Okay", $result);

      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletController::deposit');

         $walletException = self::returnWalletException($exception);
         if ($walletException != null) {
            return $walletException;
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Withdraw from wallet
    *
    * @param Request $request
    * @return Response|Application|ResponseFactory
    */
   public function withdrawal(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(WalletRules::withdraw());

      $data = $request->all();

      /** @var User $user */
      $user = auth()->user();

      try {
         //Get user wallet
         /** @var Wallet $wallet */
         $wallet = $user->{"wallet"};

         $result = $wallet->accountWithdrawal($data);

         return api_response(100, "Okay", $result);

      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'WalletController::withdrawal');
         if ($exception instanceof InsufficientWalletBalanceException) {
            return api_response(120, $exception->getMessage());
         }

         $walletException = self::returnWalletException($exception);
         if ($walletException != null) {
            return $walletException;
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Get users transactions
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function transactions(Request $request): Response|Application|ResponseFactory
   {
      $pageNumber = $request->get("page");
      $type = $request->get("type");

      /** @var User $user */
      $user = auth()->user();

      $result = $user->{"wallet"}->transactions(
         type: $type,
         pageNumber: $pageNumber,
      );

      if ($result == null) {
         return api_response(120, __("errors.unknown_error"));
      }

      return api_response(100, "Okay", $result);
   }

   // Controller utilities

   /**
    * Return exception
    *
    * @param Exception $exception
    * @return Response|Application|ResponseFactory|null
    */
   protected static function returnWalletException(Exception $exception): Response|Application|ResponseFactory|null
   {
      if ($exception instanceof WalletBlockedException || $exception instanceof WalletRemainingPinAttemptsException || $exception instanceof InvalidPasswordException || $exception instanceof WalletCurrencyChangeException) {
         return api_response(120, $exception->getMessage());
      }

      return null;
   }
}
