<?php

namespace App\Http\Controllers\Api\Platform;

use App\Exceptions\EmailAlreadyExistException;
use App\Exceptions\InvalidEmail;
use App\Exceptions\InvalidPasswordException;
use App\Exceptions\InvalidTPCodeException;
use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\UserEmail;
use App\RequestRules\Api\UserEmailRequestRule;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class UserEmailController extends Controller
{
   /**
    * Get all user phone numbers
    *
    * @return Response|Application|ResponseFactory
    */
   public function getAll(): Response|Application|ResponseFactory
   {
      /** @var User $user */
      $user = auth()->user();

      $emails = $user->{'userEmails'};

      return api_response(100, 'Okay', $emails);
   }

   /**
    * Add user email
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function add(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(UserEmailRequestRule::creationRules());

      try {
         $email = UserEmail::addEmail($request->all());

         return api_response(100, 'Ok', $email);
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserEmailController::add');

         $userEmailExceptions = self::returnUserEmailExceptions($exception);
         if ($userEmailExceptions != null) {
            return $userEmailExceptions;
         }
      }

      return api_response(101, __('errors.unknown_error'));
   }

   /**
    * Edit email
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function edit(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(UserEmailRequestRule::editRules());

      /** @var User $user */
      $user = auth()->user();

      try {
         /** @var UserEmail $emailToEdit */
         $emailToEdit = UserEmail::findByUserEmail($user->{"id"}, $request->{'old_email'});

         if ($emailToEdit == null) {
            return api_response(101, __("errors.invalid_email"));
         }

         $result = $emailToEdit->editEmail($request->all());

         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserEmailController::edit');

         $userEmailExceptions = self::returnUserEmailExceptions($exception);
         if ($userEmailExceptions != null) {
            return $userEmailExceptions;
         }
      }

      return api_response(101, __('errors.unknown_error'));
   }

   /**
    * Verify email
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function verify(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(UserEmailRequestRule::verifyRules());

      try {
         $Results = UserEmail::verify(($request->all()));

         return api_response(100, "Okay", $Results);
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserEmailController::verify');

         $userEmailExceptions = self::returnUserEmailExceptions($exception);
         if ($userEmailExceptions != null) {
            return $userEmailExceptions;
         }
      }

      return api_response(101, __('errors.unknown_error'));
   }

   /**
    * Delete email
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function delete(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(UserEmailRequestRule::deleteRules());

      /** @var User $user */
      $user = auth()->user();

      try {
         /** @var UserEmail $emailToDelete */
         $emailToDelete = UserEmail::findByUserEmail($user->{"id"}, $request->{'email'});

         if ($emailToDelete == null) {
            return api_response(101, __("errors.invalid_email"));
         }

         $emailToDelete->deleteEmail($request->all());

         return api_response(100, "Okay");
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'UserEmailController::delete');

         $userEmailExceptions = self::returnUserEmailExceptions($exception);
         if ($userEmailExceptions != null) {
            return $userEmailExceptions;
         }
      }

      return api_response(101, __('errors.unknown_error'));
   }

   // Controller utilities

   /**
    * Return exception
    *
    * @param Exception $exception
    * @return Response|Application|ResponseFactory|null
    */
   protected static function returnUserEmailExceptions(Exception $exception): Response|Application|ResponseFactory|null
   {
      if ($exception instanceof EmailAlreadyExistException || $exception instanceof InvalidEmail || $exception instanceof InvalidPasswordException || $exception instanceof InvalidTPCodeException) {
         return api_response(120, $exception->getMessage());
      }

      return null;
   }
}
