<?php

namespace App\Http\Controllers\Api\Platform;

use App\Enums\SearchHistoryTags;
use App\Http\Controllers\Controller;
use App\Models\SearchHistory;
use App\RequestRules\Api\SearchHistoryRequestRules;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class SearchHistoryController extends Controller
{
   /**
    * Get search histories
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function getHistories(Request $request): Response|Application|ResponseFactory
   {
      try {
         $histories = SearchHistory::getHistories($request->{"page"});

         if ($request->{"page"} !== null) {
            $searchResultsArray = [];
            foreach ($histories as $history) {
               $searchResultsArray[] = $history;
            }

            $result = default_paginator_format(
               $histories->lastPage(),
               $histories->total(),
               $histories->currentPage(),
               "search_histories",
               $searchResultsArray,
            );

            return api_response(100, 'Ok', $result);
         }

         return api_response(100, 'Ok', $histories);
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'SearchHistoryController::getHistories');
      }

      return api_response(101, __('errors.unknown_error'));
   }

   /**
    * Search
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function search(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(SearchHistoryRequestRules::search());

      try {
         $tag = $request->get('tag');
         $page = $request->get('page');
         $query = $request->get('query') ?? '';
         $finalData = [];

         if ($tag === null) {
            // Add query to search histories
            SearchHistory::addSearch($query);

            $usersResult = SearchHistory::searchUsers($query, $page);
            $productsResult = SearchHistory::searchProducts($query, $page);
            $servicesResult = SearchHistory::searchServices($query, $page);
            $hashtagsResult = SearchHistory::searchHashtags($query, $page);

            $results = array(
               "top" => [],
               "users" => $usersResult,
               "products" => $productsResult,
               "services" => $servicesResult,
               "hashtags" => $hashtagsResult,
               "live" => default_paginator_format(1, 0, 1, 'data', []),
            );

            $topResult = SearchHistory::computeTopSearch(
               $query,
               $usersResult['data'],
               $productsResult['data'],
               $servicesResult['data'],
               [],
               $hashtagsResult['data'],
            );
            $results['top'] = default_paginator_format(1, count($topResult), 1, 'data', $topResult);

            $finalData = $results;
         } elseif ($tag === SearchHistoryTags::users->value) {
            //Add query to search histories
            SearchHistory::addSearch($query);
            $finalData = SearchHistory::searchUsers($query, $page);
         } elseif ($tag === SearchHistoryTags::products->value) {
            //Add query to search histories
            SearchHistory::addSearch($query);
            $finalData = SearchHistory::searchProducts($query, $page);
         } elseif ($tag === SearchHistoryTags::services->value) {
            //Add query to search histories
            SearchHistory::addSearch($query);
            $finalData = SearchHistory::searchServices($query, $page);
         } elseif ($tag === SearchHistoryTags::hashtags->value) {
            //Add query to search histories
            SearchHistory::addSearch($query);
            $finalData = SearchHistory::searchHashtags($query, $page);
         } elseif ($tag === SearchHistoryTags::live->value) {
            //Add query to search histories
            SearchHistory::addSearch($query);
            $finalData = default_paginator_format(1, 0, 1, 'data', []);
         }

         return api_response(100, "Okay", $finalData);
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'SearchHistoryController::search');
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Delete search history
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function deleteHistory(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(SearchHistoryRequestRules::delete());

      try {
         SearchHistory::deleteHistory($request->{"search_history_id"});

         return api_response(100, 'Ok');
      } catch (Exception $exception) {
         log_debug(exception: $exception, prefix: 'SearchHistoryController::deleteHistory');
      }

      return api_response(120, __("errors.unknown_error"));
   }
}
