<?php

namespace App\Http\Controllers\Api\Platform;

use App\Exceptions\UploadFileException;
use App\Http\Controllers\Controller;
use App\Libs\Upload\ChunkUploadManager;
use App\Libs\Upload\UploadRealtimeMessageAttachments;
use App\Models\RealtimeMessage;
use App\RequestRules\Api\ChatRequestRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Http\UploadedFile;

class RealtimeMessageController extends Controller
{
   /**
    * Get chat message
    *
    * @param string $uuid
    * @return Response|Application|ResponseFactory
    */
   public function getMessage(string $uuid): Response|Application|ResponseFactory
   {
      /** @var ?RealtimeMessage $message */
      $message = RealtimeMessage::findByUuid($uuid);

      if ($message !== null) {
         $message = $message->serializeForChannel();

         return api_response(100, "Okay", $message);
      }

      return api_response(101);
   }

   /***
    * Send new message
    *
    * @param Request $request
    * @return Response|Application|ResponseFactory
    */
   public function newMessage(Request $request): Response|Application|ResponseFactory
   {
      // Validation ...
      $request->validate(ChatRequestRules::sendMessageRules());

      if (!$request->hasFile('attachment')) {
         /** @var ?RealtimeMessage $message */
         $message = RealtimeMessage::newMessage($request->all());

         if ($message !== null) {
            $selfSending = auth()->id() === $message->{'receiver_participant_id'};

            if (!$selfSending) {
               // Broadcast event to send message ...
               $message->broadcastMessage();

               // Send notification
               $message->sendNewMessageNotification();
            }

            return api_response(100, "Okay", $message->serializeForChannel());
         }

         return api_response(101, __("errors.unknown_error"));
      }

      try {
         $chunkManager = new ChunkUploadManager(
            requestKey: "attachment",
         );

         // Try to upload file into chunk mode
         $uploadedFile = $chunkManager->upload($request);

         if ($uploadedFile instanceof UploadedFile) {
            /** @var RealtimeMessage $message */
            $message = UploadRealtimeMessageAttachments::uploadAttachment(
               $request->all(),
               $uploadedFile,
            );

            if (filled($message)) {
               $selfSending = auth()->id() === $message->{'receiver_participant_id'};

               if (! $selfSending) {
                   // Broadcast event to send message ...
                   $message->broadcastMessage();

                   // Send notification
                   $message->sendNewMessageNotification();
               }

               return api_response(100, "Okay", $message->serializeForChannel());
            }
         }
      } catch (\Exception $exception) {
         if($exception instanceof UploadFileException) {
            return api_response(101, __("errors.unable_to_upload_file"));
         }

         log_debug($exception, "RealtimeMessageController::newMessage");
      }

      return api_response(101, __("errors.unknown_error"));
   }

   /**
    * Mark message as read
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function markAsDelivered(Request $request): Response|Application|ResponseFactory
   {
      if (filled($request->{'message_uuid'})) {
         /** @var ?RealtimeMessage $message */
         $message = RealtimeMessage::findByUuid($request->{'message_uuid'});

         $message = $message?->markAsDelivered();
         // Message broadcasting for delivery ...
         $message?->delivery();

         return api_response(100, "Okay");
      }

      return api_response(101, __("errors.unknown_error"));
   }

   /**
    * Read messages
    *
    * @param Request $request
    * @return Response|Application|ResponseFactory
    */
   public function markAsRead(Request $request): Response|Application|ResponseFactory
   {
      $uuids = $request->{'message_uuids'};

      if (filled($uuids)) {
         RealtimeMessage::markMessagesAsRead($uuids);
      }

      return api_response(100, "Okay");
   }
}
