<?php

namespace App\Http\Controllers\Api\Platform;

use App\Exceptions\ProductRatingException;
use App\Http\Controllers\Controller;
use App\Models\Product;
use App\Models\ProductRating;
use App\Models\User;
use App\RequestRules\Api\ProductRatingRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class ProductRatingController extends Controller
{
    /**
     * Create or update product rating
     *
     * @param Request $request
     * @return Response|Application|ResponseFactory
     */
    public function createOrUpdate(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(ProductRatingRules::reviewProduct());

        try {
            $rating = ProductRating::createOrUpdateReview($request->all());

            return api_response(100, "Okay", array(
                "rating" => $rating->{'product'}->averageRating(),
                "user_rating" => $rating,
            ));
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProductRatingController::createOrUpdate');

            if ($exception instanceof ProductRatingException) {
                return api_response(120, $exception->getMessage());
            }
        }

        return api_response(120, __("errors.unknown_error"));
    }

    /**
     * All product ratings history
     *
     * @param int $productId
     * @param Request $request
     * @return Response|Application|ResponseFactory
     */
    public function productRatingsHistory(int $productId, Request $request): Response|Application|ResponseFactory
    {
        /** @var ?Product $product */
        $product = Product::findById($productId);
        /** @var User $user */
        $user = auth()->user();

        if ($product === null) {
            return api_response(101, "errors.unknown_product");
        }

        $ratings = $product->allRatings($request->get('page'));
        $result = array(
            "can_rate_product" => $user->canRateProduct($product->{'id'}) ? config("torryme.constants.default_one_number") : config("torryme.constants.default_zero_number"),
            "new_product_rating" => $product->averageRating(),
            "last_user_rating" => ProductRating::lastUserRating(
                $user,
                $product->{"id"}
            )?->load(['user.userDetail', 'user.business']),
            "data" => $ratings,
        );

        return api_response(100, "Okay", $result);
    }
}
