<?php

namespace App\Http\Controllers\Api\Platform;

use App\Enums\BookUploadKeyEnum;
use App\Enums\GameUploadKeyEnum;
use App\Enums\MovieUploadKeyEnum;
use App\Enums\MusicTypeEnum;
use App\Enums\MusicUploadKeyEnum;
use App\Enums\ProductStepEnum;
use App\Enums\SeriesUploadKeyEnum;
use App\Exceptions\InvalidProductStateException;
use App\Exceptions\UploadFileException;
use App\Http\Controllers\Controller;
use App\Libs\Upload\ChunkUploadManager;
use App\Libs\Upload\UploadProductAttachments;
use App\Models\Product;
use App\Models\User;
use App\RequestRules\Api\ProductRules;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Http\UploadedFile;

class ProductController extends Controller
{
    /**
     * Add goods products
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function addGoods(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(ProductRules::createGoods());

        // Get data
        $data = $request->all();

        if ($data["step"] == ProductStepEnum::create->value) {
            $goodsProduct = Product::createGoodsProduct($data);

            if ($goodsProduct == null) {
                return api_response(120, __('errors.unknown_error'));
            }

            return api_response(100, "Okay", $goodsProduct);
        }

        if ($data["step"] == ProductStepEnum::attachments->value) {
            /** @var Product $product */
            $product = Product::findById($request->{'product_id'});

            $chunkManager = new ChunkUploadManager(
                requestKey: "attachment",
            );

            try {
                // Try to upload file into chunk mode
                $uploadedFile = $chunkManager->upload($request);

                if($uploadedFile instanceof UploadedFile) {
                    $result = UploadProductAttachments::uploadAttachments($uploadedFile, $product);

                    if(filled($result)) {
                        return api_response(100, "Okay", $product->refresh());
                    }
                }
            } catch (Exception $exception) {
                log_debug(exception: $exception,  prefix: 'ProductController::addGoods');
            }
        }

        return api_response(120, __('errors.unknown_error'));
    }

    /**
     * Create service category
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function addService(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(ProductRules::createService());

        // Get data
        $data = $request->all();

        if ($data["step"] == ProductStepEnum::create->value) {
            $serviceProduct = Product::createServiceProduct($data);

            if ($serviceProduct == null) {
                return api_response(120, __('errors.unknown_error'));
            }

            return api_response(100, "Okay", $serviceProduct);
        }

        if ($data["step"] == ProductStepEnum::attachments->value) {
            /** @var Product $product */
            $product = Product::findById($request->{'product_id'});

            $chunkManager = new ChunkUploadManager(
                requestKey: "attachment",
            );

            try {
                // Try to upload file into chunk mode
                $uploadedFile = $chunkManager->upload($request);

                if($uploadedFile instanceof UploadedFile) {
                    $result = UploadProductAttachments::uploadAttachments($uploadedFile, $product);

                    if(filled($result)) {
                        return api_response(100, "Okay", $product->refresh());
                    }
                }
            } catch (Exception $exception) {
                log_debug(exception: $exception,  prefix: 'ProductController::addGoods');
            }
        }

        return api_response(120, __('errors.unknown_error'));
    }

    /**
     * Create library product
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function addLibrary(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(ProductRules::createLibrary());

        // Get data
        $data = $request->all();

        if ($data["step"] == ProductStepEnum::create->value) {
            $libraryProduct = Product::createLibraryProduct($data);

            if ($libraryProduct == null) {
                return api_response(120, __('errors.unknown_error'));
            }

            return api_response(100, "Okay", $libraryProduct);
        }

        if ($data["step"] == ProductStepEnum::attachments->value) {
            /** @var Product $product */
            $product = Product::findById($request->{'product_id'});

            $uploadKey = "attachment";
            if($request->hasFile(BookUploadKeyEnum::cover->value)) {
                $uploadKey = BookUploadKeyEnum::cover->value;
            }
            if($request->hasFile(BookUploadKeyEnum::bookFile->value)) {
                $uploadKey = BookUploadKeyEnum::bookFile->value;
            }

            $chunkManager = new ChunkUploadManager(
                requestKey: $uploadKey,
            );

            try {
                // Try to upload file into chunk mode
                $uploadedFile = $chunkManager->upload($request);

                if($uploadedFile instanceof UploadedFile) {
                    $result = UploadProductAttachments::uploadAttachmentForLibrary($uploadedFile, $product, $uploadKey);

                    if(filled($result)) {
                        return api_response(100, "Okay", $product->refresh());
                    }
                }
            } catch (Exception $exception) {
                log_debug(exception: $exception,  prefix: 'ProductController::addGoods');

                if($exception instanceof InvalidProductStateException) {
                    return api_response(120, $exception->getMessage());
                }
            }
        }

        return api_response(120, __('errors.unknown_error'));
    }

    /**
     * Create cinema category
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function addCinema(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(ProductRules::createCinema());

        // Get data
        $data = $request->all();

        if ($data["step"] == ProductStepEnum::create->value) {
            try {
                $cinemaProduct = Product::createCinemaProduct($data);

                return api_response(100, "Okay", $cinemaProduct->load('currency'));
            } catch (Exception $exception) {
                log_debug(exception: $exception, prefix: 'ProductController::addCinema');

                if($exception instanceof UploadFileException) {
                    return api_response(101, __('errors.unable_to_upload_series_poster'));
                }
            }

            return api_response(120, __('errors.unknown_error'));
        }

        if ($data["step"] == ProductStepEnum::attachments->value) {
            /** @var Product $product */
            $product = Product::findById($request->{'product_id'});

            if($product->{'businessSubCategoryChild'}->{'code'} === config('torryme.business_sub_category_child_codes.movie')) {
                $uploadKey = "attachment";
                if($request->hasFile(MovieUploadKeyEnum::poster->value)) {
                    $uploadKey = MovieUploadKeyEnum::poster->value;
                }
                if($request->hasFile(MovieUploadKeyEnum::movieFile->value)) {
                    $uploadKey = MovieUploadKeyEnum::movieFile->value;
                }

                $chunkManager = new ChunkUploadManager(
                    requestKey: $uploadKey,
                );

                try {
                    // Try to upload file into chunk mode
                    $uploadedFile = $chunkManager->upload($request);

                    if($uploadedFile instanceof UploadedFile) {
                        $result = UploadProductAttachments::uploadAttachmentForMovie($uploadedFile, $product, $uploadKey);

                        if(filled($result)) {
                            return api_response(100, "Okay", $product->refresh());
                        }
                    }
                } catch (Exception $exception) {
                    log_debug(exception: $exception,  prefix: 'ProductController::addCinema');
                }
            }

            if($product->{'businessSubCategoryChild'}->{'code'} === config('torryme.business_sub_category_child_codes.series')) {
                // Validation
                $request->validate(array(
                    'season_occurrence' => 'required|numeric|gt:0',
                    'episode_season_occurrence' => 'required|numeric|gt:0',
                ));

                // Upload key ...
                $chunkManager = new ChunkUploadManager(
                    requestKey:  SeriesUploadKeyEnum::seasonEpisode->value,
                );

                try {
                    // Try to upload file into chunk mode
                    $uploadedFile = $chunkManager->upload($request);

                    if($uploadedFile instanceof UploadedFile) {
                        $result = UploadProductAttachments::uploadAttachmentForSeriesSeason(
                            $request->all(),
                            $uploadedFile,
                            $product,
                        );

                        if(filled($result)) {
                            return api_response(100, "Okay", $product->refresh());
                        }
                    }
                } catch (Exception $exception) {
                    log_debug(exception: $exception,  prefix: 'ProductController::addCinema');

                    if($exception instanceof InvalidProductStateException) {
                        return api_response(120, $exception->getMessage());
                    }
                }
            }
        }

        return api_response(120, __('errors.unknown_error'));
    }

    /**
     * Create music category
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function addMusic(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(ProductRules::createMusic());

        // Get all
        $data = $request->all();

        if ($data["step"] == ProductStepEnum::create->value) {
            try {
                $musicProduct = Product::createMusicProduct($data);

                return api_response(100, "Okay", $musicProduct->load('currency'));
            } catch (Exception $exception) {
                log_debug(exception: $exception, prefix: 'ProductController::addMusic');

                if($exception instanceof UploadFileException) {
                    return api_response(101, __('errors.unable_to_upload_album_poster'));
                }
            }

            return api_response(120, __('errors.unknown_error'));
        }

        if ($data["step"] == ProductStepEnum::attachments->value) {
            /** @var Product $product */
            $product = Product::findById($request->{'product_id'});

            if($data["music_type"] === MusicTypeEnum::single->value) {
                $uploadKey = "attachment";
                if($request->hasFile(MusicUploadKeyEnum::musicPoster->value)) {
                    $uploadKey = MusicUploadKeyEnum::musicPoster->value;
                }
                if($request->hasFile(MusicUploadKeyEnum::musicFile->value)) {
                    $uploadKey = MusicUploadKeyEnum::musicFile->value;
                }

                $chunkManager = new ChunkUploadManager(
                    requestKey: $uploadKey,
                );

                try {
                    // Try to upload file into chunk mode
                    $uploadedFile = $chunkManager->upload($request);

                    if($uploadedFile instanceof UploadedFile) {
                        $result = UploadProductAttachments::uploadAttachmentForMusic($uploadedFile, $product, $uploadKey);

                        if(filled($result)) {
                            return api_response(100, "Okay", $product->refresh());
                        }
                    }
                } catch (Exception $exception) {
                    log_debug(exception: $exception,  prefix: 'ProductController::addMusic');
                }
            }

            if($data["music_type"] === MusicTypeEnum::album->value) {
                // Upload key ...
                $chunkManager = new ChunkUploadManager(
                    requestKey: MusicUploadKeyEnum::musicFile->value,
                );

                try {
                    // Try to upload file into chunk mode
                    $uploadedFile = $chunkManager->upload($request);

                    if($uploadedFile instanceof UploadedFile) {
                        $result = UploadProductAttachments::uploadAttachmentForMusicAlbum(
                            $request->all(),
                            $uploadedFile,
                            $product,
                        );

                        if(filled($result)) {
                            return api_response(100, "Okay", $product->refresh());
                        }
                    }
                } catch (Exception $exception) {
                    log_debug(exception: $exception,  prefix: 'ProductController::addMusic');

                    if($exception instanceof InvalidProductStateException) {
                        return api_response(120, $exception->getMessage());
                    }
                }
            }
        }

        return api_response(120, __('errors.unknown_error'));
    }

    /**
     * Add product for game category
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function addGame(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(ProductRules::createGames());

        // Get data
        $data = $request->all();

        if ($data["step"] == ProductStepEnum::create->value) {
            try {
                $productGame = Product::addGameProduct($data);

                if ($productGame == null) {
                    return api_response(120, __('errors.unknown_error'));
                }

                return api_response(100, "Okay", $productGame);
            } catch (Exception $exception) {
                log_debug(exception: $exception, prefix: 'ProductController::addGame');

                if($exception instanceof UploadFileException) {
                    return api_response(101, __('errors.unable_to_upload_game_cover'));
                }
            }

            return api_response(120, __('errors.unknown_error'));
        }

        if ($data["step"] == ProductStepEnum::attachments->value) {
            /** @var Product $product */
            $product = Product::findById($request->{'product_id'});

            $uploadKey = "attachment";
            if($request->hasFile(MusicUploadKeyEnum::musicPoster->value)) {
                $uploadKey = GameUploadKeyEnum::gameCover->value;
            }
            if($request->hasFile(MusicUploadKeyEnum::musicFile->value)) {
                $uploadKey = GameUploadKeyEnum::gameFile->value;
            }

            $chunkManager = new ChunkUploadManager(
                requestKey: $uploadKey,
            );

            try {
                // Try to upload file into chunk mode
                $uploadedFile = $chunkManager->upload($request);

                if($uploadedFile instanceof UploadedFile) {
                    $result = UploadProductAttachments::uploadAttachmentForGame($uploadedFile, $product, $uploadKey);

                    if(filled($result)) {
                        return api_response(100, "Okay", $product->refresh());
                    }
                }
            } catch (Exception $exception) {
                log_debug(exception: $exception,  prefix: 'ProductController::addGame');
            }
        }

        return api_response(120, __('errors.unknown_error'));
    }

    /**
     * Update product status to complete
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function updateStatusToComplete(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(ProductRules::updateProductStatus());
        /** @var Product $productStatus */
        $productStatus = Product::findById($request->get('product_id'));

        $productStatus == $productStatus->updateStatusToComplete();

        // Try to load series relation ...
        if ($productStatus->{'businessSubCategoryChild'}->{'code'} === config('torryme.business_sub_category_child_codes.series')) {
            $productStatus = $productStatus->load(['series.seasons', 'currency']);
        }

        return api_response(100, "Okay", $productStatus);
    }

    /**
     * Compute product details for visitor
     *
     * @param int $productId
     * @return Response|Application|ResponseFactory
     */
    public function productDetailsForVisitor(int $productId): Response|Application|ResponseFactory
    {
        /** @var Product $product */
        $product = Product::findById($productId);

        if($product === null) {
            return api_response(101, __('errors.unknown_product'));
        }

        try {
            $data = $product->computeDetailsForVisitor();

            if($data !== null) {
                return api_response(100, "Okay", $data);
            }

            return api_response(100, __('errors.unknown_product'));
        }catch (\Exception $exception) {
            log_debug($exception, "ProductController::productDetailsForVisitor");
        }

        return api_response(101, __('errors.unknown_error'));
    }

    /**
     * Get similar product of another product
     *
     * @param int $productId
     * @param Request $request
     * @return Response|Application|ResponseFactory
     */
    public function similarProductsToAnother(int $productId, Request $request): Response|Application|ResponseFactory
    {
        /** @var Product $product */
        $product = Product::findById($productId);

        if($product === null) {
            return api_response(100, "Okay", []);
        }

        try {
            $data = $product->similarProductsToAnother([], $request->{'page'});

            return api_response(100, "Okay", $data);
        }catch (\Exception $exception) {
            log_debug($exception, "ProductController::similarProductsToAnother");
        }

        return api_response(101, __('errors.unknown_error'));
    }

    /**
     * Get business product sample
     *
     * @param Request $request
     * @return Response|Application|ResponseFactory
     */
    public function sampleProducts(Request $request): Response|Application|ResponseFactory
    {
        /** @var User $user */
        $user = auth()->user();

        try {
            $data = Product::businessProductSample($user->{'business'}, $request->{'page'});

            return api_response(100, "Okay", $data);
        }catch (\Exception $exception) {
            log_debug($exception, "ProductController::sampleProducts");
        }

        return api_response(101, __('errors.unknown_error'));
    }
}
