<?php

namespace App\Http\Controllers\Api\Platform;

use App\Events\EmailVerificationEvent;
use App\Events\TelephoneVerificationEvent;
use App\Events\TwoFactorAuthCodeEvent;
use App\Http\Controllers\Api\Auth\AuthenticateController;
use App\Http\Controllers\Controller;
use App\Models\OtpVerification;
use App\Models\User;
use App\Models\UserTelephone;
use App\RequestRules\Api\OtpVerificationRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Validation\ValidationException;

class OtpVerificationController extends Controller
{
   /**
    * Send otp code request
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    * @throws ValidationException
    */
   public function sendOtpCode(Request $request): Response|Application|ResponseFactory
   {
      $this->validate($request, OtpVerificationRules::sendOtpRules());

      // Check if telephone exist as a secondary number
      if (UserTelephone::checkIsUsedAsSecondaryNumber($request->{'telephone'})) {
         return api_response(150, __('errors.telephone_already_exist'));
      }

      // Send telephone verification code
      event(new TelephoneVerificationEvent(
         $request->{'telephone'},
         otp_code(),
      ));

      return api_response(100, "Okay");
   }

   /**
    * Checking otp code
    *
    * @param Request $request
    * @return Response|Application|ResponseFactory
    */
   public function checkOtp(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(OtpVerificationRules::checkingOtpRules());

      try {
         /** @var OtpVerification $otpEntry */
         $otpEntry = OtpVerification::findByEmailOrTelephone($request->{'telephone'}, null);

         if ($otpEntry === null || $otpEntry->{'otp'} !== $request->{'otp'}) {
            return api_response(101, __('errors.invalid_otp_code'));
         }

         if ($otpEntry->expired()) {
            return api_response(101, __('errors.expired_otp_code'));
         }

         // Update otp verification
         $otpEntry->markedAsValidate();

         // Check user and last device ...
         /** @var User $user */
         $user = User::oneStepAuthentication($request->{'telephone'});

         return api_response(
            100, 'Okay',
            $user !== null ? AuthenticateController::responseWithUser($user) : null,
         );
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'OtpVerificationController::checkOtp');
      }

      return api_response(103, __('errors.unknown_error'));
   }

   /**
    * Send verification code
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function sendForVerification(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(OtpVerificationRules::sendOtpRules(withEmail:  true));

      try {
         if (!filled($request->{'telephone'}) && !filled($request->{'email'})) {
            return api_response(150, __('errors.credentials_error'));
         }

         // Send telephone verification code
         if (filled($request->{'telephone'})) {
            //Check telephone number uniqueness
            User::checkPhoneNumberUniqueness($request->{'telephone'});
            event(new TelephoneVerificationEvent(
               $request->{'telephone'},
               otp_code(),
            ));
         }

         // Send email verification code
         if (filled($request->{'email'})) {
            event(new EmailVerificationEvent(
               $request->{'email'},
               otp_code(),
            ));
         }
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'OtpVerificationController::sendVerification');

         return api_response(120, $exception->getMessage());
      }

      return api_response(100, "Okay");
   }

   /**
    * Verify otp code
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function singleVerification(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(OtpVerificationRules::checkingOtpRules(withEmail: true));

      if (!filled($request->{'telephone'}) && !filled($request->{'email'})) {
         return api_response(150, __('errors.credentials_error'));
      }

      try {
         /** @var OtpVerification $otpEntry */
         $otpEntry = OtpVerification::findByEmailOrTelephone(
            $request->{'telephone'},
            $request->{'email'}
         );

         if ($otpEntry === null) {
            return api_response(101, __('errors.invalid_otp_code'));
         }

         if ($otpEntry->expired()) {
            return api_response(101, __('errors.expired_otp_code'));
         }

         if ($otpEntry->{'otp'} !== $request->{'otp'}) {
            $attemptsLeft = $otpEntry->decrementAttempt();

            if ($attemptsLeft === config("torryme.constants.default_zero_number")) {
               if (filled($request->{'telephone'})) {
                  // Send telephone verification code
                  event(new TelephoneVerificationEvent(
                     $request->{'telephone'},
                     otp_code(),
                  ));
               }

               if (filled($request->{'email'})) {
                  // Send email verification code
                  event(new EmailVerificationEvent(
                     $request->{'email'},
                     otp_code(),
                  ));
               }

               // Reset otp remaining attempts
               $otpEntry->resetRemainingOtpAttempts();
            }

            return api_response(101, __('errors.invalid_otp_code'));
         }

         // Update otp verification
         $otpEntry->markedAsValidate();

         // Reset otp remaining attempts
         $otpEntry->resetRemainingOtpAttempts();

         return api_response(
            100,
            'Okay',
            $otpEntry->{'user'} !== null ? AuthenticateController::responseWithUser($otpEntry->{'user'}) : null,
         );
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'OtpVerificationController::checkOtp');
      }

      return api_response(103, __('errors.unknown_error'));
   }

   /**
    * Send 2factor auth code
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function send2FactorAuthCode(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(['email' => 'required|email']);

      try {
         /** @var User $user */
         $user =  auth()->user();

         // Send confidential code
         if (filled($request->{'email'})) {
            event(new TwoFactorAuthCodeEvent(
               $request->{'email'},
               otp_code(),
               $user
            ));
         }
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'OtpVerificationController::send2FactorAuthCode');
         return api_response(120, $exception->getMessage());
      }

      return api_response(100, "Okay");
   }

   /**
    * Verify 2factor auth code
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function verify2FactorAuthCode(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(OtpVerificationRules::checking2FactorsCodeRules());

      /** @var User $user */
      $user =  auth()->user();

      try {
         /** @var OtpVerification $otpEntry */
         $otpEntry = OtpVerification::findByEmailOrTelephone(null, $request->{'email'});

         if ($otpEntry === null) {
            return api_response(101, __('errors.invalid_otp_code'));
         }

         if ($otpEntry->expired()) {
            return api_response(101, __('errors.expired_otp_code'));
         }

         if ($otpEntry->{'otp'} !== $request->{'otp'}) {
            $attemptsLeft = $otpEntry->decrementAttempt();

            if ($attemptsLeft === config("torryme.constants.default_zero_number")) {
               event(new TwoFactorAuthCodeEvent(
                  $request->{'email'},
                  otp_code(),
                  $user,
               ));

               // Reset otp remaining attempts
               $otpEntry->resetRemainingOtpAttempts();
            }

            return api_response(101, __('errors.invalid_otp_code'));
         }

         // Update otp verification
         $otpEntry->markedAsValidate();

         // Reset otp remaining attempts
         $otpEntry->resetRemainingOtpAttempts();

         return api_response(
            100,
            'Okay',
            $otpEntry->{'user'} !== null ? AuthenticateController::responseWithUser($otpEntry->{'user'}) : null,
         );
      } catch (\Exception $exception) {
         log_debug(exception: $exception, prefix: 'OtpVerificationController::verify2FactorAuthCode');
      }

      return api_response(103, __('errors.unknown_error'));
   }
}
