<?php

namespace App\Http\Controllers\Api\Platform;

use App\Exceptions\InvalidCouponExpiryDate;
use App\Exceptions\InvalidCouponStartDate;
use App\Http\Controllers\Controller;
use App\Models\Coupon;
use App\RequestRules\Api\CouponRequestRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class CouponController extends Controller
{
    /**
     * Get discount types
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function discountTypes(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(CouponRequestRules::getTypes());
        try {
            $finalData = Coupon::computeCouponTypesFromPaginator(
                Coupon::discountTypes(
                    $request->get('discount_type') ?? null,
                    $request->get('business_id'),
                    $request->get('page') ?? null,
                )
            );

            return api_response(100, "Okay", $finalData);
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'CouponController::discountTypes');
        }

        return api_response(120, __("errors.unknown_error"));
    }

    /**
     * Create coupon
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function create(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(CouponRequestRules::create());
        $data = $request->all();
        try {
             Coupon::create($data);

            return api_response(100, 'Ok', Coupon::statistics());
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'CouponController::create');
            if ($exception instanceof InvalidCouponExpiryDate || $exception instanceof InvalidCouponStartDate) {
                return api_response(120, $exception->getMessage());
            }
        }

        return api_response(101, __('errors.unknown_error'));
    }

    /**
     * Coupon statistics
     *
     * @return Application|ResponseFactory|Response
     */
    public function statistics(): Response|Application|ResponseFactory
    {
        try {
            $couponStatistics = Coupon::statistics();

            return api_response(100, 'Ok', $couponStatistics);
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'CouponController::statistics');
        }

        return api_response(101, __('errors.unknown_error'));
    }

    public function computeCouponFees(Request $request) {

    }
}
