<?php

namespace App\Http\Controllers\Api\Platform;

use App\Enums\GenericStatusEnum;
use App\Enums\PostStepEnum;
use App\Exceptions\CommunityNameAlreadyTaken;
use App\Exceptions\DoubleMembershipRequestException;
use App\Exceptions\InvalidAddCommunityAddSubscriberException;
use App\Exceptions\InvalidCommunityMembershipCancellationAttempt;
use App\Exceptions\InvalidCommunityMembershipDeclinationAttempt;
use App\Exceptions\InvalidCommunityMembershipValidationAttempt;
use App\Exceptions\InvalidGetCommunityReportAttempt;
use App\Exceptions\InvalidMembershipRequestAttempt;
use App\Exceptions\InvalidSubscriberAction;
use App\Exceptions\InvalidSuspensionAction;
use App\Exceptions\UploadFileException;
use App\Http\Controllers\Controller;
use App\Libs\Upload\ChunkUploadManager;
use App\Libs\Upload\UploadPostAttachments;
use App\Models\Community;
use App\Models\CommunityMembershipRequest;
use App\Models\CommunityReport;
use App\Models\CommunitySubscriber;
use App\Models\CommunitySuspension;
use App\Models\Post;
use App\Models\User;
use App\RequestRules\Api\CommunityRequestRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Http\UploadedFile;

class CommunityController extends Controller
{
   /**
    * Create community
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function create(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::create());

      try {
         $community = Community::createCommunity($request->all());
         $community = $community->buildCommonSerialization();

         return api_response(100, "ok", $community);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::create");

         if ($exception instanceof UploadFileException) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(101, __('errors.unknown_error'));
   }

   /**
    * Edit community
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function edit(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::edit());

      try {
         $data = $request->all();

         /** @var Community $community */
         $community = Community::findById($data["community_id"]);

         $community = $community->editCommunity($data);
         $community = $community->buildCommonSerialization();

         return api_response(100, "Okay", $community);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::edit");

         if ($exception instanceof CommunityNameAlreadyTaken) {
            return api_response(120, $exception->getMessage());
         }

         if ($exception instanceof UploadFileException) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Edit community setting
    *
    * @param Request $request
    * @param int $communityId
    * @return Application|ResponseFactory|Response
    */
   public function editSetting(Request $request, int $communityId): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::editSetting());

      try {
         /** @var Community $community */
         $community = Community::findById($communityId);

         if ($community == null) {
            return api_response(101, __("errors.unknown_community"));
         }

         $result = $community->editOrUpdateSettings($request->all());

         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::editSetting");
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Delete community
    *
    * @param int $communityId
    * @return Application|ResponseFactory|Response
    */
   public function deleteCommunity(int $communityId): Response|Application|ResponseFactory
   {
      /** @var ?Community $community */
      $community = Community::findById($communityId);

      if ($community == null) {
         return api_response(101, __("errors.unknown_community"));
      }

      return api_response(100, "Okay", $community->deleteCommunity());
   }

   /**
    * Create report community
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function addReport(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::addReport());

      try {
         $data = $request->all();
         /** @var Community $community */
         $community = Community::findById($data['community_id']);

         $results = $community->report($data);

         return api_response(100, "ok", $results);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::addReport");
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Get community reports
    *
    * @param int $communityId
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function getCommunityReports(int $communityId, Request $request): Response|Application|ResponseFactory
   {
      try {
         /** @var ?Community $community */
         $community = Community::findById($communityId);

         if ($community == null) {
            return api_response(101, __("errors.unknown_community"));
         }

         $result = CommunityReport::getCommunityReports(
            $communityId,
            $request->get("page")
         );

         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::getCommunityReports");

         if ($exception instanceof InvalidGetCommunityReportAttempt) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Delete community subscriber
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function deleteSubscriber(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::marksAsAdminOrMemberOrDeleteMember());
      try {
         $data = $request->all();

         /** @var CommunitySubscriber $communitySubscriber */
         $communitySubscriber = CommunitySubscriber::getCommunitySubscriber($data["user_id"], $data["community_id"]);

         if ($communitySubscriber == null) {
            return api_response(101, __("errors.unknown_subscriber"));
         }

         return api_response(100, "Okay", $communitySubscriber->deleteSubscriber($data));
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::deleteSubscriber");
         if ($exception instanceof InvalidSubscriberAction) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Leave community
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function leaveCommunity(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::leaveCommunity());

      try {
         $result = CommunitySubscriber::leaveCommunity($request->{"community_id"});
         $result->load('user.userDetail', 'user.business');

         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::leaveCommunity");

         if ($exception instanceof InvalidSubscriberAction) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Add community suspension
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function addSuspension(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::addSuspension());

      $data = $request->all();
      $result = CommunitySuspension::addSuspension($data);
      if ($result == null) {
         return api_response(120, __("errors.unknown_error"));
      }

      return api_response(100, 'Ok', $result);
   }

   /**
    * Remove community suspension
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function removeSuspension(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::removeSuspension());
      try {
         $data = $request->all();

         /** @var CommunitySuspension $communitySuspension */
         $communitySuspension = CommunitySuspension::findSuspension($data["user_id"], $data["community_id"]);

         if ($communitySuspension == null) {
            return api_response(101, __("errors.unknown_error"));
         }

         return api_response(100, "Okay", $communitySuspension->removeSuspension($data));
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::removeSuspension");
         if ($exception instanceof InvalidSuspensionAction) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Mark as Admin subscriber
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function markAsAdmin(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::marksAsAdminOrMemberOrDeleteMember());

      try {
         $data = $request->all();
         $data["action"] = config("torryme.community_subscriber_action.mark_as_admin");

         /** @var CommunitySubscriber $isCommunityMember */
         $isCommunityMember = CommunitySubscriber::checkIsCommunityAdminOrAlreadySubscriber(
            GenericStatusEnum::disable->value,
            $data["user_id"],
            $data["community_id"]
         );

         if ($isCommunityMember == null) {
            return api_response(101, __("errors.unknown_subscriber"));
         }

         $result = $isCommunityMember->markAsAdminOrMember($data);
         $result->load('user.userDetail', 'user.business');

         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::markAsAdmin");

         if ($exception instanceof InvalidSubscriberAction) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Mark Admin as member subscriber
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function markAsMember(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::marksAsAdminOrMemberOrDeleteMember());

      try {
         $data = $request->all();
         $data["action"] = config("torryme.community_subscriber_action.mark_as_member");

         /** @var CommunitySubscriber $isCommunityAdmin */
         $isCommunityAdmin = CommunitySubscriber::checkIsCommunityAdminOrAlreadySubscriber(
            GenericStatusEnum::enable->value,
            $data["user_id"],
            $data["community_id"]
         );

         if ($isCommunityAdmin == null) {
            return api_response(101, __("errors.unknown_subscriber"));
         }

         $result = $isCommunityAdmin->markAsAdminOrMember($data);
         $result->load('user.userDetail', 'user.business');

         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::markAsAdmin");

         if ($exception instanceof InvalidSubscriberAction) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Get community post feeds
    *
    * @param int $communityId
    * @param Request $request
    * @return Response|Application|ResponseFactory
    */
   public function getCommunityFeeds(int $communityId, Request $request): Response|Application|ResponseFactory
   {
      try {
         /** @var ?Community $community */
         $community = Community::findById($communityId);

         if ($community == null) {
            return api_response(101, __("errors.unknown_community"));
         }

         $communityPostFeeds = $community->postsFeed($request->{'page'});

         return api_response(100, "Okay", Post::buildPostsPaginator($communityPostFeeds));
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::getCommunityFeeds");
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Republish community post
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function republishCommunityPost(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::repostRules());

      try {
         $postId = $request->{'post_id'};

         /** @var Post $post */
         $post = Post::findById($postId);
         $post = $post->republishCommunityPost();

         return api_response(100, "Okay", $post->commonSerialization());
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::republishCommunityPost");
      }

      return api_response(120, __('errors.unknown_error'));
   }

   /**
    * Add subscriber to community
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function addSubscriber(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::addSubscriber());

      try {
         $result = Community::addSubscriberToCommunity($request->all());
         $result?->load('user.userDetail', 'user.business');

         return api_response(100, 'Ok', $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::addSubscriber");
         if ($exception instanceof InvalidAddCommunityAddSubscriberException) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Get user's communities
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function usersCommunities(Request $request): Response|Application|ResponseFactory
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         $communities = Community::getUserCommunities(
            $user->{"id"},
            $request->get("page")
         );
         return api_response(100, 'Ok', $communities);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::usersCommunities");
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Get community suggestions
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function communitySuggestions(Request $request): Response|Application|ResponseFactory
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         $communities = Community::communitySuggestions($user->{"id"}, $request->get("page"));

         return api_response(100, 'Ok', $communities);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::usersCommunities");
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Add membership request
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function addMembershipRequest(Request $request): Response|Application|ResponseFactory
   {
      $request->validate(CommunityRequestRules::addMembershipRequest());

      try {
         $membershipRequest = CommunityMembershipRequest::addMembershipRequest(
            $request->all()
         );
         $membershipRequest->load('user.userDetail', 'user.business');

         return api_response(100, 'Ok', $membershipRequest);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::addMembershipRequest");

         if ($exception instanceof DoubleMembershipRequestException) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Cancel membership request
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function cancelMembershipRequest(Request $request): Response|Application|ResponseFactory
   {
      try {
         /** @var CommunityMembershipRequest $membershipRequest */
         $membershipRequest = CommunityMembershipRequest::findById(
            $request->{"community_membership_request_id"},
         );

         if ($membershipRequest == null) {
            return api_response(101, __("errors.unknown_membership_request"));
         }

         $result = $membershipRequest->cancelMembershipRequest();
         $result->load('user.userDetail', 'user.business');

         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::cancelMembershipRequest");

         if ($exception instanceof InvalidCommunityMembershipCancellationAttempt) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Decline membership request
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function declineMembershipRequest(Request $request): Response|Application|ResponseFactory
   {
      try {
         /** @var CommunityMembershipRequest $membershipRequest */
         $membershipRequest = CommunityMembershipRequest::findById(
            $request->{"community_membership_request_id"}
         );

         if ($membershipRequest == null) {
            return api_response(101, __("errors.unknown_membership_request"));
         }

         $result = $membershipRequest->declineMembershipRequest();
         $result->load('user.userDetail', 'user.business');

         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::declineMembershipRequest");

         if ($exception instanceof InvalidCommunityMembershipDeclinationAttempt) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Approve membership request
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function approveMembershipRequest(Request $request): Response|Application|ResponseFactory
   {
      try {
         /** @var CommunityMembershipRequest $membershipRequest */
         $membershipRequest = CommunityMembershipRequest::findById(
            $request->{"community_membership_request_id"}
         );

         if ($membershipRequest == null) {
            return api_response(101, __("errors.unknown_membership_request"));
         }

         $result = $membershipRequest->approveMembershipRequest();

         // Notify
         $membershipRequest->{'user'}->notifyForCommunityMemberShipStatus(
            $membershipRequest->refresh()
         );

         return api_response(100, "Okay", array(
            'membership_request' => $membershipRequest->refresh()->load('user.userDetail', 'user.business'),
            'community_member' => $result->load('user.userDetail', 'user.business')
         ));
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::approveMembershipRequest");

         if ($exception instanceof InvalidCommunityMembershipValidationAttempt) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Get membership requests
    *
    * @param int $communityId
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function memberRequests(int $communityId, Request $request): Response|Application|ResponseFactory
   {
      try {
         /** @var ?Community $community */
         $community = Community::findById($communityId);

         if ($community == null) {
            return api_response(101, __("errors.unknown_community"));
         }

         $result = CommunityMembershipRequest::memberRequests($communityId, $request->get("page"));
         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::memberRequests");

         if ($exception instanceof InvalidMembershipRequestAttempt) {
            return api_response(120, $exception->getMessage());
         }
      }

      return api_response(120, __("errors.unknown_error"));
   }

   /**
    * Get users membership requests
    *
    * @param Request $request
    * @return Application|ResponseFactory|Response
    */
   public function userRequests(Request $request): Response|Application|ResponseFactory
   {
      try {
         /** @var User $user */
         $user = auth()->user();

         $result = CommunityMembershipRequest::getUserMembershipRequests($user->{"id"}, $request->get("page"));
         return api_response(100, "Okay", $result);
      } catch (\Exception $exception) {
         log_debug($exception, "CommunityController::userRequests");
      }

      return api_response(120, __("errors.unknown_error"));
   }

}
