<?php

namespace App\Http\Controllers\Api\Platform;

use App\Exceptions\BusinessCreationException;
use App\Http\Controllers\Controller;
use App\Models\Business;
use App\Models\Post;
use App\Models\Product;
use App\Models\User;
use App\RequestRules\Api\BusinessRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;

class BusinessController extends Controller
{
    /**
     * Create business
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function create(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(BusinessRules::create());

        $data = $request->all();

        try {
            $business = Business::createBusiness($data);

            return api_response(100, "Okay", $business);
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'BusinessController::create');

            if ($exception instanceof BusinessCreationException) {
                return api_response(120, $exception->getMessage());
            }
        }

        return api_response(120, __("errors.unknown_error"));
    }

    /**
     * We try to calculate and return the products available in the store of a business
     *
     * @param Request $request
     * @return Response|Application|ResponseFactory
     */
    public function productsShop(Request $request): Response|Application|ResponseFactory
    {
        /** @var Business $business */
        $business = $request->{'business_id'} !== null ? Business::findById($request->{'business_id'}) : null;

        if($business === null) {
            return api_response(101, __('errors.unknown_business'));
        }

        try {
            $data = Product::productsForBusinessShop($business, $request->{'page'});

            return api_response(100, "Okay", $data);
        }catch (\Exception $exception) {
            log_debug($exception, "BusinessController::productsForBusinessShop");
        }

        return api_response(101, __('errors.unknown_error'));
    }

    /**
     * We try to calculate and return the products
     * available in the store of a business for visitor
     *
     * @param Request $request
     * @return Response|Application|ResponseFactory
     */
    public function productForVisitorShop(Request $request): Response|Application|ResponseFactory
    {
        /** @var Business $business */
        $business = $request->{'business_id'} !== null ? Business::findById($request->{'business_id'}) : null;

        if($business === null) {
            return api_response(101, __('errors.unknown_business'));
        }

        try {
            $data = Product::productsForVisitorBusinessShop($business, $request->{'page'});

            return api_response(100, "Okay", $data);
        }catch (\Exception $exception) {
            log_debug($exception, "BusinessController::productForVisitorShop");
        }

        return api_response(101, __('errors.unknown_error'));
    }

    /**
     * Get business posts
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function getBusinessPosts(Request $request): Response|Application|ResponseFactory
    {
        /** @var User $user */
        $user = auth()->user();

        try {
            $posts = Post::getUsersPost($user, $request->get("page"));

            return api_response(100, "Okay", $posts);
        } catch (\Exception $exception) {
            log_debug($exception, "BusinessController::posts");
        }

        return api_response(120, __("errors.unknown_error"));
    }
}
