<?php

namespace App\Http\Controllers\Api\Platform;

use App\Exceptions\AlreadyAffiliatedToProgramException;
use App\Exceptions\ShouldBeAffiliateMarketerException;
use App\Http\Controllers\Controller;
use App\Models\BusinessAffiliate;
use App\Models\User;
use App\RequestRules\Api\BusinessAffiliateRules;
use Exception;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Carbon;

class BusinessAffiliateController extends Controller
{
    /**
     * Affiliate to program
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function affiliateToProgram(Request $request): Response|Application|ResponseFactory
    {
        $request->validate(BusinessAffiliateRules::affiliateToProgram());

        try {
            /** @var User $user */
            $user = auth()->user();
            $affiliatedBusiness = $user->{"business"};

            $businessAffiliate = BusinessAffiliate::affiliateToProgram(
                $request->get('affiliate_program_id'),
                $affiliatedBusiness,
            );

            return api_response(100, "Okay", $businessAffiliate);
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'BusinessAffiliateController::affiliateToProgram');
            $businessAffiliateExceptions = self::BusinessAffiliateExceptions($exception);
            if ($businessAffiliateExceptions != null) {
                return $businessAffiliateExceptions;
            }
        }

        return api_response(120, __("errors.unknown_error"));
    }

    /**
     * Get affiliated programs
     *
     * @param Request $request
     * @return Application|ResponseFactory|Response
     */
    public function affiliatedPrograms(Request $request): Response|Application|ResponseFactory
    {
        try {
            /** @var User $user */
            $user = auth()->user();
            $affiliatedBusiness = $user->{"business"};

            $affiliatedPrograms = BusinessAffiliate::getAffiliatedPrograms($affiliatedBusiness, $request->get("page"));

            $affiliatedProgramsArray = array();
            foreach ($affiliatedPrograms as $affiliatedProgram) {
                $business = $affiliatedProgram->{"businessAffiliate"};

                $affiliatedProgramsArray[] = array(
                    "business_affiliation_id" => $affiliatedProgram->{"id"},
                    "affiliate_program" => $affiliatedProgram->{"affiliateProgram"},
                    "affiliate_program_business_id" => $business->{"id"},
                    "affiliate_program_business_designation" => $business->{"designation"},
                    "affiliate_program_business_owner_photo" => $business->{"ownerUser"}->{"avatar"},
                    "created_at" => Carbon::parse($affiliatedProgram->{'created_at'})->utc(),
                    "expired_at" => Carbon::parse($affiliatedProgram->{'expired_at'})->utc(),
                );
            }

            $finalData = default_paginator_format(
                $affiliatedPrograms->lastPage(),
                $affiliatedPrograms->total(),
                $affiliatedPrograms->currentPage(),
                'business_affiliations',
                $affiliatedProgramsArray
            );

            return api_response(100, "Okay", $finalData);
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'BusinessAffiliateController::affiliatedPrograms');

            $businessAffiliateExceptions = self::businessAffiliateExceptions($exception);
            if ($businessAffiliateExceptions != null) {
                return $businessAffiliateExceptions;
            }
        }

        return api_response(120, __("errors.unknown_error"));
    }

    // Controller utilities

    /**
     * Return exceptions
     *
     * @param Exception $exception
     * @return Application|ResponseFactory|Response|null
     */
    protected static function businessAffiliateExceptions(Exception $exception): Response|Application|ResponseFactory|null
    {
        if ($exception instanceof AlreadyAffiliatedToProgramException || $exception instanceof ShouldBeAffiliateMarketerException) {
            return api_response(120, $exception->getMessage());
        }

        return null;
    }
}
