<?php

namespace App\Services;

use App\Models\Specialty;
use App\Models\SpecialtyClass;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\HigherOrderBuilderProxy;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait SpecialtyService
{

    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Specialty::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all specialties
     *
     * @return Builder[]|Collection
     */
    public static function allSpecialties(): Collection|array
    {
        return Specialty::query()->get();
    }

    /**
     * Create specialty
     *
     * @param string $designation
     * @param array $classIds
     * @return Specialty|Builder|Model|null
     */
    public static function createSpecialty(string $designation, array $classIds): Model|Builder|Specialty|null
    {
        DB::beginTransaction();

        try {
            /** @var Specialty $result */
            $specialty = self::store([
                "designation" => $designation,
            ]);

            if ($specialty != null) {
                $specialty->classes()->attach($classIds);
                $result = $specialty;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SpecialtyService::createSpecialty');
            DB::rollBack();
            $result = null;
        }

        return $result;
    }

    /**
     * Find specialty by id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function findById(int $id): Model|Builder|null
    {

        return Specialty::query()->where("id", $id)->first();
    }

    /**
     * Update specialty with given id
     *
     * @param int $id
     * @param string $designation
     * @param array $classIds
     * @return Model|Builder|null
     */
    public static function updateSpecialty(int $id, string $designation, array $classIds): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            Specialty::query()->where('id', $id)->update(["designation" => $designation]);

            $specialty = self::findById($id);

            if ($specialty != null) {
                $specialty->classes()->sync($classIds);
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SpecialtyService::updateSpecialty');
            DB::rollBack();
            $specialty = null;
        }

        return $specialty;
    }

    /**
     * Delete specialty with given id
     *
     * @param int $id
     * @return Specialty|null
     */
    public static function deleteSpecialty(int $id): ?Specialty
    {
        DB::beginTransaction();
        try {

            /** @var Specialty $specialty */

            $specialty = self::findById($id);

            if ($specialty != null) {
                $specialty->classes()->detach();
                $specialty->delete();
                DB::commit();
            }
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SpecialtyService::deleteSpecialty');
            DB::rollBack();
            $specialty = null;
        }

        return $specialty;
    }

    /**
     * Get specialtyClass id
     *
     * @param int $specialtyId
     * @param int $ClassId
     * @return mixed
     */
    public static function getSpecialtyClassId(int $specialtyId, int $ClassId): mixed
    {
        $specialtyClass = SpecialtyClass::query()->where('specialty_id', $specialtyId)->where('class_id', $ClassId)->first();

        return $specialtyClass->{"id"};
    }
}
