<?php

namespace App\Services;

use App\Jobs\ConvertLessonForStreaming;
use App\Models\Lesson;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use ProtoneMedia\LaravelFFMpeg\Support\FFMpeg;

trait LessonService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Lesson::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all lessons
     *
     * @return Builder[]|Collection
     */
    public static function allLessons(): Collection|array
    {

        return Lesson::query()->get();
    }

    /**
     * Find lesson with given id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function findById(int $id): Model|Builder|null
    {
        return Lesson::query()->where('id', $id)->first();
    }

    /**
     * Create lesson for general education
     *
     * @param array $data
     * @param string $educationTypeId
     * @param string $type
     * @return Lesson|Builder|Model|null
     */
    public static function createLessonGeneralEducation(array $data, string $educationTypeId, string $type): Model|Lesson|Builder|null
    {
        DB::beginTransaction();
        try {
            $path = null;
            $thumbnailPath = null;
            $lesson = null;

            $path = self::handleVideoGeneralEducation($data);
            $thumbnailPath = self::handleThumbnailGeneration($data, $type);
            $lessonLengthInSeconds = FFMpeg::open($data['lesson_video'])->getDurationInSeconds();

            if($lessonLengthInSeconds >= config('minesec.lesson_length.one_hour') ) {
                $duration =  gmdate("H:i:s",$lessonLengthInSeconds);
            } else {
                $duration =  gmdate("i:s",$lessonLengthInSeconds);
            }

            /** @var Lesson $results */

            foreach ($data['classIds'] as $classId) {
                $lesson = self::store([
                    'sub_system_id' => $data['sub_system_id'],
                    'education_type_id' => $educationTypeId,
                    'class_id' => $classId,
                    'subject_id' => $data['subjectIds'][0],
                    'term_id' => $data['term_id'],
                    'title' => $data['lesson_title'],
                    'description' => $data['lesson_description'],
                    'duration' => $duration,
                    'type' => $type,
                    'path' => $path,
                    'thumbnail' => $thumbnailPath,
                ]);

//                ConvertLessonForStreaming::dispatch($lesson);
            }

            DB::commit();
            $results = $lesson;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonService::createLessonGeneralEducation');
            self::deleteVideo($path);
            self::deleteThumbnail($thumbnailPath);
            DB::rollBack();

            $results = null;
        }

        return $results;
    }

    /**
     * Create lesson for technical education
     *
     * @param array $data
     * @param string $educationTypeId
     * @param string $type
     * @return Lesson|Builder|Model|null
     */
    public static function createLessonTechnicalEducation(array $data, string $educationTypeId, string $type): Model|Lesson|Builder|null
    {
        DB::beginTransaction();
        try {
            $lesson = null;
            $path = null;
            $thumbnailPath = null;

            $path = self::handleVideTechnicalEducation($data);
            $thumbnailPath = self::handleThumbnailGeneration($data, $type);

            $lessonLengthInSeconds = FFMpeg::open($data['lesson_video'])->getDurationInSeconds();

            if($lessonLengthInSeconds >= config('minesec.lesson_length.one_hour') ) {
                $duration =  gmdate("H:i:s",$lessonLengthInSeconds);
            } else {
                $duration =  gmdate("i:s",$lessonLengthInSeconds);
            }

            /** @var Lesson $results */;

            foreach ($data['specialtyClassIds'] as $classId) {
                $lesson = self::store([
                    'sub_system_id' => $data['sub_system_id'],
                    'education_type_id' => $educationTypeId,
                    'sub_education_type_id' => $data['sub_education_type_id'],
                    'specialty_id' => $data['specialty_id'],
                    'class_id' => $classId,
                    'subject_id' => $data['subjectIds'][0],
                    'term_id' => $data['term_id'],
                    'title' => $data['lesson_title'],
                    'description' => $data['lesson_description'],
                    'duration' => $duration,
                    'type' => $type,
                    'path' => $path,
                    'thumbnail' => $thumbnailPath,
                ]);
            }

            ConvertLessonForStreaming::dispatch($results);
            DB::commit();

            $results = $lesson;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonService::createLessonTechnicalEducation');
            self::deleteVideo($path);
            self::deleteThumbnail($thumbnailPath);
            DB::rollBack();

            $results = null;
        }

        return $results;
    }

    /**
     * Create lesson for teacher training
     *
     * @param array $data
     * @param string $educationTypeId
     * @param string $type
     * @return Lesson|Builder|Model|null
     */
    public static function createLessonTeacherTraining(array $data, string $educationTypeId, string $type): Model|Lesson|Builder|null
    {
        DB::beginTransaction();
        try {
            $thumbnailPath = null;
            $path = null;
            $lesson = null;

            $path = self::handleVideTeacherTraining($data);
            $thumbnailPath = self::handleThumbnailGeneration($data, $type);

            $lessonLengthInSeconds = FFMpeg::open($data['lesson_video'])->getDurationInSeconds();

            if($lessonLengthInSeconds >= config('minesec.lesson_length.one_hour') ) {
                $duration =  gmdate("H:i:s",$lessonLengthInSeconds);
            } else {
                $duration =  gmdate("i:s",$lessonLengthInSeconds);
            }

            /** @var Lesson $results */

            foreach ($data['leveClassIds'] as $class) {
                $lesson = self::store([
                    'sub_system_id' => $data['sub_system_id'],
                    'education_type_id' => $educationTypeId,
                    'level_id' => $data['level_id'],
                    'class_id' => $class,
                    'subject_id' => $data['subjectIds'][0],
                    'term_id' => $data['term_id'],
                    'title' => $data['lesson_title'],
                    'description' => $data['lesson_description'],
                    'duration' => $duration,
                    'type' => $type,
                    'path' => $path,
                    'thumbnail' => $thumbnailPath,
                ]);
            }
            DB::commit();

            $results = $lesson;
        } catch (Exception $exception)  {
            log_debug(exception: $exception, prefix: 'LessonService::createLessonTeacherTraining');
            self::deleteVideo($path);
            self::deleteThumbnail($thumbnailPath);
            DB::rollBack();

            $results = null;
        }

        return $results;
    }

    /**
     * @param $filename
     * @return string
     */
    private static function getCleanFileName($filename): string
    {
        return  str_replace('.mp4',  '', str_replace(' ', '-', strtolower($filename))).'.png';
    }

    /**s
     * Handle thumbnail capture and  uploading
     *
     * @param array $data
     * @param string $type
     * @return string|null
     */
    public static function handleThumbnailGeneration(array $data, string $type): ?string
    {
        if($type == config('minesec.education_type.general')) {

            $subSystem = sub_system($data['sub_system_id']);
            $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
            $claas = classes($data['classIds'][0]);
            $subject = subject($data['subjectIds'][0]);

            $thumbNailName =  self::getCleanFileName($data['lesson_video']->getClientOriginalName());

            FFMpeg::open($data['lesson_video'])->getFrameFromSeconds(config('minesec.jingle_length.length'))->export()
                ->toDisk('public')
                ->save('lesson_thumbnails/'.$subSystem.'/'.$educationType. '/'.$claas.'/'.$subject . '/' . $thumbNailName);

            return 'lesson_thumbnails/'.$subSystem.'/'.$educationType. '/'.$claas.'/'.$subject . '/' . $thumbNailName;
        }

        if($type == config('minesec.education_type.technical')) {

            $subject = subject($data['subjectIds'][0]);
            $subSystem = sub_system($data['sub_system_id']);
            $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
            $subEducationType = sub_education_type($data['sub_education_type_id']);
            $specialty = specialty($data['specialty_id']);
            $claas = classes($data['specialtyClassIds'][0]);

            $thumbNailName =  self::getCleanFileName($data['lesson_video']->getClientOriginalName());

            FFMpeg::open($data['lesson_video'])->getFrameFromSeconds(config('minesec.jingle_length.length'))->export()
                ->toDisk('public')
                ->save('lesson_thumbnails/'.$subSystem.'/'.$educationType.'/'.$subEducationType . '/'. $specialty .'/'.$claas.'/'.$subject . '/' . $thumbNailName);

            return 'lesson_thumbnails/'.$subSystem.'/'.$educationType.'/'.$subEducationType . '/'. $specialty .'/'.$claas.'/'.$subject . '/' . $thumbNailName;
        }

        if($type == config('minesec.education_type.teacher_training')) {

            $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
            $level = level($data['level_id']);
            $claas = classes($data['leveClassIds'][0]);
            $subject = subject($data['subjectIds'][0]);
            $subSystem = sub_system($data['sub_system_id']);

            $thumbNailName =  self::getCleanFileName($data['lesson_video']->getClientOriginalName());

            FFMpeg::open($data['lesson_video'])->getFrameFromSeconds(config('minesec.jingle_length.length'))->export()
                ->toDisk('public')
                ->save('lesson_thumbnails/'.$subSystem.'/'.$educationType.'/'.$level . '/'.$claas.'/'.$subject . '/' . 'FrameAt10sec.png');

            return 'lesson_thumbnails/'.$subSystem.'/'.$educationType.'/'.$level . '/'.$claas.'/'.$subject . '/' . $thumbNailName;
        }

        return null;
    }

    /**
     * Upload video for general education
     *
     * @param array $data
     * @return mixed
     */
    public static function handleVideoGeneralEducation( array $data): mixed
    {
        $subSystem = sub_system($data['sub_system_id']);
        $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
        $claas = classes($data['classIds'][0]);
        $subject = subject($data['subjectIds'][0]);

        $file = $data['lesson_video'];
        $name = $file->getClientOriginalName();

        return $file->storeAs('lessons/'.$subSystem.'/'.$educationType.'/'.$claas.'/'.$subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Upload video for technical education
     *
     * @param array $data
     * @return mixed
     */
    public static function  handleVideTechnicalEducation( array $data): mixed
    {
        $specialty = specialty($data['specialty_id']);
        $subSystem = sub_system($data['sub_system_id']);
        $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
        $subEducationType = sub_education_type($data['sub_education_type_id']);
        $claas = classes($data['specialtyClassIds'][0]);
        $subject = subject($data['subjectIds'][0]);

        $file = $data['lesson_video'];
        $name = $file->getClientOriginalName();

        return $file->storeAs('lessons/'.$subSystem.'/'.$educationType.'/'.$subEducationType . '/'.$specialty . '/'.$claas.'/'.$subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Upload video for teacher training
     *
     * @param array $data
     * @return mixed
     */
    public static function  handleVideTeacherTraining( array $data): mixed
    {
        $subSystem = sub_system($data['sub_system_id']);
        $educationType = education_type(preg_replace("/[^0-9]/", "", $data['education_type_id']));
        $level = level($data['level_id']);
        $claas = classes($data['leveClassIds'][0]);
        $subject = subject($data['subjectIds'][0]);

        $file = $data['lesson_video'];
        $name = $file->getClientOriginalName();

        return $file->storeAs('lessons/'.$subSystem.'/'.$educationType.'/'.$level . '/'.$claas.'/'.$subject, strtolower(str_replace(' ', '-', $name)), 'public');
    }

    /**
     * Delete lesson with given id
     *
     * @param int $id
     * @return Lesson|null
     */
    public static function deleteLesson(int $id): ?Lesson
    {
        DB::beginTransaction();
        try {
            /** @var Lesson $results */

            $lesson = self::findById($id);
            $path = $lesson->{"path"};
            $thumbnail = $lesson->{"thumbnail"};
            self::deleteVideo($path);
            self::deleteThumbnail($thumbnail);
            self::destroy($id);
            $results = $lesson;
            DB::commit();
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'LessonService::deleteLesson');
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Delete video in given path
     *
     * @param string $path
     * @return bool
     */
    public static  function deleteVideo(string $path): bool
    {
        return Storage::disk('public')->delete($path);

    }

    /**
     * @param string $thumbNailPath
     * @return bool
     */
    public static function deleteThumbnail(string $thumbNailPath): bool
    {
        return Storage::disk('public')->delete($thumbNailPath);
    }

    /**
     * Get subject lessons
     *
     * @param array $data
     * @return Builder[]|Collection
     */
    public static function subjectLessons(array $data): Collection|array
    {
        $lessons = Lesson::query();

        if($data["type"] == config('minesec.education_type.general')) {
            $lessons->whereNull(array('sub_education_type_id', 'specialty_id', 'level_id'))
                    ->where('type', $data["type"])
                    ->where('sub_system_id', $data["sub_system_id"])
                    ->where('education_type_id', $data["education_type_id"])
                    ->where('class_id', $data["class_id"])
                    ->where('subject_id', $data["subject_id"]);
        }

        if($data["type"] == config('minesec.education_type.technical')) {
            $lessons->whereNull('level_id')
                ->where('type', $data["type"])
                ->where('sub_system_id', $data["sub_system_id"])
                ->where('education_type_id', $data["education_type_id"])
                ->where('sub_education_type_id', $data["sub_education_type_id"])
                ->where('specialty_id', $data["specialty_id"])
                ->where('class_id', $data["class_id"])
                ->where('subject_id', $data["subject_id"]);
        }

        if($data["type"] == config('minesec.education_type.teacher_training')) {
            $lessons->whereNull(array('sub_education_type_id', 'specialty_id'))
                ->where('type', $data["type"])
                ->where('sub_system_id', $data["sub_system_id"])
                ->where('education_type_id', $data["education_type_id"])
                ->where('level_id', $data["level_id"])
                ->where('class_id', $data["class_id"])
                ->where('subject_id', $data["subject_id"]);
        }

        return $lessons->get();
    }

}
