<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Models\EducationType;
use App\Models\SubSystem;
use App\RequestRules\Web\SubSystemRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class SubSystemController extends Controller
{
    /**
     * Return sub system view
     *
     * @return Application|Factory|View
     */
    public function index(): View|Factory|Application
    {

        $educationTypes = EducationType::allEducationTypes();
        $subSystems = SubSystem::allSubSystems();

        return view("admin.sub_system.index", compact("subSystems", "educationTypes"));
    }

    /**
     * Create sub system
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {
        $request->validate(SubSystemRules::create());

        $designation = $request->{"designation"};
        $educationTypeIds = $request->{"educationTypeIds"};

        $response = SubSystem::createSubSystem($designation, $educationTypeIds);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('sub-system.index')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Get sub system with given id
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function edit(Request $request): View|Factory|Application
    {
        $request->validate(SubSystemRules::edit($request->merge(["id" => $request->route("id")])));
        $id = $request->{"id"};

        $subSystems = SubSystem::allSubSystems();
        $subSystem = SubSystem::findById($id);
        $educationTypes = EducationType::allEducationTypes();

        return view("admin.sub_system.edit", compact("educationTypes", "subSystems", "subSystem"));
    }

    /**
     * Update sub system with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(SubSystemRules::update());

        $id = $request->{"id"};
        $designation = $request->{"designation"};
        $educationTypeIds = $request->{"educationTypeIds"};

        $response = SubSystem::updateSubsystem($id, $designation, $educationTypeIds);

        if($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('sub-system.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete sub system with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(SubSystemRules::delete());

        $id = $request->{"id"};

        $response = SubSystem::deleteSubSystem($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }
}
