<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Models\EducationType;
use App\Models\Specialty;
use App\Models\SubEducationType;
use App\RequestRules\Web\SubEducationTypeRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class SubEducationTypeController extends Controller
{

    /**
     * Get Sub education type list view
     *
     * @return Application|Factory|View
     */
    public  function index(): View|Factory|Application
    {

        $subEducationTypes = SubEducationType::allSubEducationTypes();
        $specialties = Specialty::allSpecialties();

        return view("admin.sub_education_type.index", compact("subEducationTypes", "specialties"));
    }

    /**
     * Create sub education type
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {
        $request->validate(SubEducationTypeRules::create());

        $designation = $request->{"designation"};
        $specialtyIds = $request->{"specialtyIds"};

        $response = SubEducationType::createSubEducationType($designation, $specialtyIds);

        if($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('sub-education-type.index')->with("success", __("messages.record_successfully_created"));

    }

    /**
     * Get sub education type with given id
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public static function edit(Request $request): View|Factory|Application
    {
        $request->validate(SubEducationTypeRules::edit($request->merge(["id" => $request->route("id")])));

        $id = $request->{"id"};

        $specialties = Specialty::allSpecialties();
        $subEducationType = SubEducationType::findById($id);
        $subEducationTypes = SubEducationType::allSubEducationTypes();

        return view("admin.sub_education_type.edit", compact("specialties", "subEducationType", "subEducationTypes"));

    }

    /**
     * Update sub education type with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(SubEducationTypeRules::create());

        $id = $request->{"id"};
        $designation = $request->{"designation"};
        $specialtyIds = $request->{"specialtyIds"};

        $response = SubEducationType::updateSubEducationType($id, $designation, $specialtyIds);

        if($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('sub-education-type.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete sub education type with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(SubEducationTypeRules::delete());

        $id = $request->{"id"};

        $response = SubEducationType::deleteSubEducationType($id);


        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Get sub education types
     *
     * @param Request $request
     * @return JsonResponse
     */
    public  function getSubEducationTypes(Request $request): JsonResponse
    {

        $request->validate(SubEducationTypeRules::getSubEducationTypes());

        $id = $request->{"education_type_id"};

        $EducationType = EducationType::findById($id);

        $subEducationTypes = array();
        foreach ($EducationType->{"subEducation"} as $subEducationType) {
            $subEducationTypes[] =array(
                "id" => $subEducationType->{"id"},
                "designation" => $subEducationType->{"designation"}
            );
        }

        return response()->json($subEducationTypes);
    }
}
