<?php

namespace App\Services;

use App\Models\User;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Hash;

trait UserService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return User::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * set and/or change language locale
     *
     * @param string $locale
     * @param string|null $requestType
     * @return string
     */
    public static function setLanguage(string $locale, string $requestType = null): string
    {
        /** @var User $user */
        $user = auth()->user();

        app()->setLocale($locale);
        session()->put('locale', $locale);

        if ($requestType != null && $requestType == config('minesec.request_type.change_locale')) {

            return User::query()->where("id", $user->{"id"})->update(["locale" => $locale]);
        }

        return $locale;
    }

    /**
     * Find by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return User::query()->find($id);
    }

    /**
     * Get personnel
     *
     * @return Builder[]|Collection
     */
    public static function getInspectoratePersonnel(): Collection|array
    {
        $inspectorateId = app("inspectorateId");

        $inspectoratePersonnel = User::query()->with("inspectorate");

        if($inspectorateId != null) {

            return $inspectoratePersonnel->where("inspectorate_id", $inspectorateId)->get();
        }

        return $inspectoratePersonnel->whereNotNull("inspectorate_id")->get();
    }

    /**
     * Create inspectorate personnel
     *
     * @param array $data
     * @return User|Builder|Model|null
     */
    public static function createInspectoratePersonnel(array $data): Model|Builder|User|null
    {
        try {
            /** @var User $results */

            $inspectoratePersonnel = self::store([
                "inspectorate_id" => $data["inspectorate_id"],
                "first_name" => $data["first_name"],
                "last_name" => $data["last_name"],
                "phone_number" => $data["phone_number"],
                "password" => Hash::make($data["phone_number"]),
                "gender" => $data["gender"],
                "locale" => $data["locale"],
                "active" => config("minesec.constants.active"),
                "verified_at" =>now(),
                "blocked_at" => null,
            ]);

            $inspectoratePersonnel->assignRole($data["role"]);

            $results = $inspectoratePersonnel;

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'UserService::createInspectoratePersonnel');
            $results = null;
        }

        return $results;
    }

    /**
     * Update personnel with given id
     *
     * @param array $data
     * @return bool|int|null
     */
    public static function updateInspectoratePersonnel(array $data): bool|int|null
    {
        try {

            $personnel = self::findById($data["id"]);

            $personnelToUpdate = $personnel->update([
                "first_name" => $data["first_name"],
                "last_name" => $data["last_name"],
                "phone_number" => $data["phone_number"],
                "password" =>  $data["password"] == null ? $personnel->{"password"} : Hash::make($data["phone_number"]),
                "gender" => $data["gender"],
                "locale" => $data["locale"],
            ]);

            $personnel->syncRoles($data["role"]);

            $results = $personnelToUpdate;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'UserService::updateInspectoratePersonnel');
            $results = null;
        }

        return $results;
    }

    /**
     * Delete user
     *
     * @param int $id
     * @return int|null
     */
    public static function deleteUser(int $id): ?int
    {
        try {
            $user = User::destroy($id);
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'UserService::deleteUser');
            $user = null;
        }
        return $user;
    }

    /**Block user
     *
     * @param int $id
     * @return bool|int|null
     */
    public static function blockUser(int $id): bool|int|null
    {
        try {
            $user = self::findById($id);
            $blockUser = $user->update(["blocked_at" => now()]);
        }catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'UserService::blockUser');
            $blockUser = null;
        }

        return $blockUser;
    }
}
