<?php

namespace App\Services;

use App\Models\SubEducationType;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait SubEducationTypeService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return SubEducationType::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get Sub education types
     *
     * @return Builder[]|Collection
     */
    public static function allSubEducationTypes(): Collection|array
    {
        return SubEducationType::query()->get();
    }

    /**
     * Create sub education type
     *
     * @param string $designation
     * @param array $specialtyIds
     * @return SubEducationType|Builder|Model|null
     */
    public static function createSubEducationType(string $designation, array $specialtyIds): Model|Builder|SubEducationType|null
    {

        DB::beginTransaction();

        try {
            /** @var SubEducationType $result */
            $subEducationType = self::store([
                "designation" => $designation,
            ]);

            if ($subEducationType != null) {
                $subEducationType->specialties()->attach($specialtyIds);
                $result = $subEducationType;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubEducationTypeService::createSubEducationType');
            DB::rollBack();
            $result = null;
        }

        return $result;
    }

    /**
     * Find sub education type with given ud
     *
     * @param $id
     * @return Model|Builder|null
     */
    public static function findById($id): Model|Builder|null
    {
        return SubEducationType::query()->where("id", $id)->first();
    }

    /**
     * Update the sub education type with the given id
     *
     * @param int $id
     * @param string $designation
     * @param array $specialtyIds
     * @return Model|Builder|null
     */
    public static  function updateSubEducationType(int $id, string $designation, array $specialtyIds): Model|Builder|null
    {

        DB::beginTransaction();

        try {
        SubEducationType::query()->where('id', $id)->update(["designation" => $designation]);

            $subEducationType = self::findById($id);

            if ($subEducationType != null) {
                $subEducationType->specialties()->sync($specialtyIds);
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubEducationTypeService::updateSubEducationType');
            DB::rollBack();
            $subEducationType = null;
        }

        return $subEducationType;
    }

    /**
     * Delete sub education type with given id
     *
     * @param int $id
     * @return SubEducationType|null
     */
    public static function deleteSubEducationType(int $id): ?SubEducationType
    {
        DB::beginTransaction();
        try {

            /** @var SubEducationType $subEducationType */

            $subEducationType = self::findById($id);

            if($subEducationType != null) {
                $subEducationType->specialties()->detach();
                $subEducationType->delete();
                DB::commit();
            }
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubEducationTypeService::deleteSubEducationType');
            DB::rollBack();
            $subEducationType = null;
        }

        return $subEducationType;
    }
}
