<?php

namespace App\Services;

use App\Models\DigitisationType;
use App\Models\DigitizationTool;
use App\Models\LessonsDigitised;
use App\Models\ProgramCoverage;
use App\Models\ProgramCoverageMeasure;
use App\Models\School;
use App\Models\SeminarTopic;
use App\Models\User;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;

trait ProgramCoverageService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return ProgramCoverage::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Create general education program coverage
     *
     *
     * @param array $data
     * @param int $educationTypeId
     * @return Builder|Model|null
     */
    public static function createGeneralEducationProgramCoverage(array $data, int $educationTypeId): Model|Builder|null
    {
        /** @var User $user */
        $user = auth()->user();
        $userName = $user->{"first_name"}." ".$user->{"last_name"};
        $schoolName = self::schoolName($data["school_id"]);

        $percentageNumberOfLessonsDone = (($data["number_of_lessons_done"]/$data["number_of_lessons_programmed"]) * 100);
        $percentageNumberOfHoursTaught = (($data["number_of_hours_taught"]/$data["number_of_teaching_hours"]) * 100);
        $percentageNumberOfLessonsDigitalised = (($data["number_of_lessons_digitalised"]/$data["number_of_lessons_to_digitalised"]) * 100);
        DB::beginTransaction();
        try {
            $programCoverage = self::store([
                "school" => $schoolName,
                "school_id" => $data["school_id"],
                "inspector_name" =>  $userName,
                "inspector_user_id" =>  $user->{"id"},

                "sub_system_id" => $data["sub_system_id"],
                "education_type_id" => $educationTypeId,
                "class_id" => $data["class_id"],
                "subject_id" => $data["subject_id"],

                "number_of_lessons_programmed" => $data["number_of_lessons_programmed"],
                "number_of_lessons_done" => $data["number_of_lessons_done"],
                "percentage_number_of_lessons_done" => $percentageNumberOfLessonsDone,
                "comment_number_of_lessons_done" => $data["comment_number_of_lessons_done"],

                "number_of_teaching_hours" => $data["number_of_teaching_hours"],
                "number_of_hours_taught" => $data["number_of_hours_taught"],
                "percentage_number_of_hours_taught" => $percentageNumberOfHoursTaught,
                "comment_number_of_hours_taught" => $data["comment_number_of_hours_taught"],

                "number_of_lessons_to_digitalised" => $data["number_of_lessons_to_digitalised"],
                "number_of_lessons_digitalised" => $data["number_of_lessons_digitalised"],
                "percentage_number_of_lessons_digitalised" => $percentageNumberOfLessonsDigitalised,
                "comment_number_of_lessons_digitalised" => $data["comment_number_of_lessons_digitalised"],

                "ots_effective" => $data["ots_effective"],
                "number_of_ots_participants" => $data["number_of_ots_participants"],
                "number_of_seminars_organised" => $data["number_of_seminars_organised"],
                "number_of_seminar_participants" => $data["number_of_seminar_participants"],
            ]);

            self::handleOtherTables($data, $programCoverage->{"id"});

            DB::commit();
            $results = $programCoverage;
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProgramCoverageService::createProgramCoverage');

            DB::rollBack();

            $results = null;
        }

        return $results;
    }

    /**
     * Create technical education program coverage
     *
     * @param array $data
     * @param int $educationTypeId
     * @return Model|Builder|null
     */
    public static function createTechEducationProgramCoverage(array $data, int $educationTypeId): Model|Builder|null
    {
        /** @var User $user */
        $user = auth()->user();
        $userName = $user->{"first_name"}." ".$user->{"last_name"};
        $schoolName = self::schoolName($data["school_id"]);

        $percentageNumberOfLessonsDigitalised = (($data["number_of_lessons_digitalised"]/$data["number_of_lessons_to_digitalised"]) * 100);
        $percentageNumberOfLessonsDone = (($data["number_of_lessons_done"]/$data["number_of_lessons_programmed"]) * 100);
        $percentageNumberOfHoursTaught = (($data["number_of_hours_taught"]/$data["number_of_teaching_hours"]) * 100);

        DB::beginTransaction();
        try {
            $programCoverage = self::store([
                "school" => $schoolName,
                "school_id" => $data["school_id"],
                "inspector_name" =>  $userName,
                "inspector_user_id" =>  $user->{"id"},

                "sub_system_id" => $data["sub_system_id"],
                "education_type_id" => $educationTypeId,
                "sub_education_type_id" => $data["sub_education_type_id"],
                "specialty_id" => $data["specialty_id"],
                "class_id" => $data["specialty_class_id"],
                "subject_id" => $data["subject_id"],

                "number_of_lessons_programmed" => $data["number_of_lessons_programmed"],
                "number_of_lessons_done" => $data["number_of_lessons_done"],
                "percentage_number_of_lessons_done" => $percentageNumberOfLessonsDone,
                "comment_number_of_lessons_done" => $data["comment_number_of_lessons_done"],

                "number_of_teaching_hours" => $data["number_of_teaching_hours"],
                "number_of_hours_taught" => $data["number_of_hours_taught"],
                "percentage_number_of_hours_taught" => $percentageNumberOfHoursTaught,
                "comment_number_of_hours_taught" => $data["comment_number_of_hours_taught"],

                "number_of_lessons_to_digitalised" => $data["number_of_lessons_to_digitalised"],
                "number_of_lessons_digitalised" => $data["number_of_lessons_digitalised"],
                "percentage_number_of_lessons_digitalised" => $percentageNumberOfLessonsDigitalised,
                "comment_number_of_lessons_digitalised" => $data["comment_number_of_lessons_digitalised"],

                "ots_effective" => $data["ots_effective"],
                "number_of_ots_participants" => $data["number_of_ots_participants"],
                "number_of_seminars_organised" => $data["number_of_seminars_organised"],
                "number_of_seminar_participants" => $data["number_of_seminar_participants"],
            ]);

            self::handleOtherTables($data, $programCoverage->{"id"});

            DB::commit();
            $results = $programCoverage;
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProgramCoverageService::createProgramCoverage');

            DB::rollBack();

            $results = null;
        }

        return $results;
    }

    /**
     * Create Teach Training education program coverage
     *
     * @param array $data
     * @param int $educationTypeId
     * @return Model|Builder|null
     */
    public static function createTeachTrainingEducationProgramCoverage(array $data, int $educationTypeId): Model|Builder|null
    {
        /** @var User $user */
        $user = auth()->user();
        $userName = $user->{"first_name"}." ".$user->{"last_name"};
        $schoolName = self::schoolName($data["school_id"]);

        $percentageNumberOfLessonsDone = (($data["number_of_lessons_done"]/$data["number_of_lessons_programmed"]) * 100);
        $percentageNumberOfLessonsDigitalised = (($data["number_of_lessons_digitalised"]/$data["number_of_lessons_to_digitalised"]) * 100);
        $percentageNumberOfHoursTaught = (($data["number_of_hours_taught"]/$data["number_of_teaching_hours"]) * 100);

        DB::beginTransaction();
        try {
            $programCoverage = self::store([
                "school" => $schoolName,
                "school_id" => $data["school_id"],
                "inspector_name" =>  $userName,
                "inspector_user_id" =>  $user->{"id"},

                "sub_system_id" => $data["sub_system_id"],
                "education_type_id" => $educationTypeId,
                "level_id" => $data["level_id"],
                "class_id" => $data["level_class_id"],
                "subject_id" => $data["subject_id"],

                "number_of_lessons_programmed" => $data["number_of_lessons_programmed"],
                "number_of_lessons_done" => $data["number_of_lessons_done"],
                "percentage_number_of_lessons_done" => $percentageNumberOfLessonsDone,
                "comment_number_of_lessons_done" => $data["comment_number_of_lessons_done"],

                "number_of_teaching_hours" => $data["number_of_teaching_hours"],
                "number_of_hours_taught" => $data["number_of_hours_taught"],
                "percentage_number_of_hours_taught" => $percentageNumberOfHoursTaught,
                "comment_number_of_hours_taught" => $data["comment_number_of_hours_taught"],

                "number_of_lessons_to_digitalised" => $data["number_of_lessons_to_digitalised"],
                "number_of_lessons_digitalised" => $data["number_of_lessons_digitalised"],
                "percentage_number_of_lessons_digitalised" => $percentageNumberOfLessonsDigitalised,
                "comment_number_of_lessons_digitalised" => $data["comment_number_of_lessons_digitalised"],

                "ots_effective" => $data["ots_effective"],
                "number_of_ots_participants" => $data["number_of_ots_participants"],
                "number_of_seminars_organised" => $data["number_of_seminars_organised"],
                "number_of_seminar_participants" => $data["number_of_seminar_participants"],
            ]);

            self::handleOtherTables($data, $programCoverage->{"id"});

            DB::commit();
            $results = $programCoverage;
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'ProgramCoverageService::createProgramCoverage');

            DB::rollBack();

            $results = null;
        }

        return $results;
    }

    /**
     * Handle other tables
     *
     * @param array $data
     * @param int $programCoverageId
     */
    public static function handleOtherTables(array $data, int $programCoverageId) {

        if(isset($data["titles_of_digitalised_lessons"]) && sizeof($data["titles_of_digitalised_lessons"]) != 0) {
            foreach ($data["titles_of_digitalised_lessons"] as $lessonTitle) {
                LessonsDigitised::store([
                    "lesson" => $lessonTitle,
                    "program_coverage_id" => $programCoverageId,
                ]);
            }
        }

        if(isset($data["digitisation_types"]) && sizeof($data["digitisation_types"]) != 0) {
            foreach ($data["digitisation_types"] as $digitisationType) {
                DigitisationType::store([
                    "type" => $digitisationType,
                    "program_coverage_id" => $programCoverageId,
                ]);
            }
        }

        if(isset($data["digitisation_tools_used"]) && sizeof($data["digitisation_tools_used"]) != 0) {
            foreach ($data["digitisation_tools_used"] as $tools) {
                DigitizationTool::store([
                    "designation" => $tools,
                    "program_coverage_id" => $programCoverageId,
                ]);
            }
        }

        if(isset($data["measures_taken_to_ensure_program_coverage"]) && sizeof($data["measures_taken_to_ensure_program_coverage"]) != 0) {
            foreach ($data["measures_taken_to_ensure_program_coverage"] as $measure) {
                ProgramCoverageMeasure::store([
                    "measure" => $measure,
                    "program_coverage_id" => $programCoverageId,
                ]);
            }
        }

        if(isset($data["seminar_topics_discussed"]) && sizeof($data["seminar_topics_discussed"]) != 0) {
            foreach ($data["seminar_topics_discussed"] as $topic) {
                SeminarTopic::store([
                    "topic" => $topic,
                    "program_coverage_id" => $programCoverageId,
                ]);
            }
        }

    }

    /**
     * Get school name
     *
     * @param int $schoolId
     * @return mixed
     */
    public static function schoolName(int $schoolId): mixed
    {

        $school = School::findById($schoolId);

        return $school->{"designation"};
    }

}
