<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Libraries\ConstantsLib;
use App\Models\EducationType;
use App\Models\ProgramCoverage;
use App\Models\Revision;
use App\Models\School;
use App\Models\SubSystem;
use App\RequestRules\Web\HomeRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class HomeController extends Controller
{

    /**
     * @var ConstantsLib
     */
    protected ConstantsLib $ConstantsLib;

    public function __construct(ConstantsLib $ConstantsLib)
    {
        $this->ConstantsLib = $ConstantsLib;
    }

    /**
     * Get home view
     *
     * @return Application|Factory|View
     */
    public function index(): View|Factory|Application
    {
        $subSystems = SubSystem::allSubSystems();
        $revisions = Revision::allRevisions();

        return view('welcome', compact("subSystems", "revisions"));
    }

    /**
     * Get progression view
     *
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function progression(): \Illuminate\Foundation\Application|View|Factory|Application
    {

        return view('progression_sheet');
    }

    /**
     * Save data collection
     *
     * @param Request $request
     * @return Application|Factory|View|\Illuminate\Foundation\Application|JsonResponse
     */
    public function dataCollection(Request $request): \Illuminate\Foundation\Application|View|Factory|JsonResponse|Application
    {
        $digitisationTypes = $this->ConstantsLib->getDigitisationTypes();
        $digitisationToolsUsed = $this->ConstantsLib->getDigitisationToolsUsed();
        $subSystems = SubSystem::allSubSystems();

        if ($request->method() == "GET") {
            return view('data_collection', compact("digitisationToolsUsed","digitisationTypes", "subSystems"));
        }

        $data = $request->all();

        $educationTypeId = preg_replace("/[^0-9]/", "", $data['education_type_id']);
        $type = EducationType::getEducationType((int)$educationTypeId);

        if ($type != null && $type == config('minesec.education_type.general')) {
            $request->validate(HomeRules::createGenEducProgramCoverage());
            $response = ProgramCoverage::createGeneralEducationProgramCoverage($data, $educationTypeId);

            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_created")]);
        }

        if ($type != null && $type == config('minesec.education_type.technical')) {
            $request->validate(HomeRules::createTechEducProgramCoverage());

            $response = ProgramCoverage::createTechEducationProgramCoverage($data, $educationTypeId);

            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_created")]);
        }

        if ($type != null && $type == config('minesec.education_type.teacher_training')) {
            $request->validate(HomeRules::createTeachTrainProgramCoverage());

            $response = ProgramCoverage::createTeachTrainingEducationProgramCoverage($data, $educationTypeId);

            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_created")]);
        }

        return response()->json(["error" => __("error.generic_error")]);
    }

    /**
     * Get data collection view
     *
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function dataCollectionOne(): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $digitisationTypes = $this->ConstantsLib->getDigitisationTypes();
        $digitisationToolsUsed = $this->ConstantsLib->getDigitisationToolsUsed();
        $subSystems = SubSystem::allSubSystems();
        $schools = School::allSchools();

        return view("form", compact("digitisationToolsUsed","digitisationTypes", "subSystems", "schools"));
    }

    public function eLibrary() {

        return view("library");
    }

    public function productDetails() {

        return view("product_details");
    }

}
