<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Libraries\ConstantsLib;
use App\Models\Revision;
use App\Models\SubSystem;
use App\RequestRules\Web\RevisionRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class RevisionController extends Controller
{

    /**
     * @var ConstantsLib
     */
    protected ConstantsLib $ConstantsLib;

    public function __construct(ConstantsLib $ConstantsLib)
    {
        $this->ConstantsLib = $ConstantsLib;
    }

    /**
     * Get revision view
     *
     * @param Request $request
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function index(Request $request): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $studios = $this->ConstantsLib->getStudios();
        $xamTypes = $this->ConstantsLib->getExamTypes();
        $subSystems = SubSystem::allSubSystems();
        $revisions = Revision::allRevisions();

        return view("admin.revision.list", compact("subSystems", "revisions", "studios", "xamTypes"));
    }

    /**
     * Create revision
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {
        $request->validate(RevisionRules::create());
        $data = $request->all();

        $response = Revision::createRevision($data);

        if ($response == null) {
            toastr()->error(__("error.generic_error"));
            return Redirect::back();
        }

        toastr()->success(__("messages.record_successfully_created"));
        return Redirect::route('revision.index');
    }

    /**
     * Get edit view
     *
     * @param Request $request
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(Request $request): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $request->validate(RevisionRules::edit($request->merge(["id" => $request->route("id")])));

        $studios = $this->ConstantsLib->getStudios();
        $xamTypes = $this->ConstantsLib->getExamTypes();
        $subSystems = SubSystem::allSubSystems();
        $revisions = Revision::allRevisions();
        $revision = Revision::findById($request->{"id"});

        return view("admin.revision.edit", compact("subSystems", "revisions", "revision", "studios", "xamTypes"));
    }

    /**
     * Update revision
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(RevisionRules::update());

        /** @var Revision $revision */
        $revision = Revision::findById($request->{"revision_id"});

        if($revision == null) {
            toastr()->error(__("error.generic_error"));
            return Redirect::back();
        }

        $response = $revision->UpdateRevision($request->all());

        if ($response == null) {
            toastr()->error(__("error.generic_error"));
            return Redirect::back();
        }

        toastr()->success(__("messages.record_successfully_updated"));
        return Redirect::route('revision.index');
    }

    /**
     * Delete revision
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(RevisionRules::delete());

        /** @var Revision $revision */
        $revision = Revision::findById($request->{"id"});

        if($revision == null) {
            return response()->json(["error" => __("error.unknown_revision")]);
        }

        $response = $revision->deleteRevision();

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

}
