<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Models\Classes;
use App\Models\EducationType;
use App\Models\Level;
use App\RequestRules\Web\LevelRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class LevelController extends Controller
{

    /**
     * Get all levels view
     *
     * @return Application|Factory|View
     */
    public  function index(): View|Factory|Application
    {
        $classes = Classes::allClasses();
        $levels = Level::allLevels();

        return view("admin.level.index", compact("classes", "levels"));
    }

    /**
     * Create level
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public  function create(Request $request): RedirectResponse
    {
        $request->validate(LevelRules::create());

        $designation = $request->{"designation"};
        $classIds = $request->{"classIds"};

        $response = Level::createLevel($designation, $classIds);

        if($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('level.index')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Get edit view
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function edit(Request $request): View|Factory|Application
    {
        $request->validate(LevelRules::edit($request->merge(["id" => $request->route("id")])));
        $id = $request->{"id"};

        $level = Level::findById($id);
        $levels = Level::allLevels();
        $classes = Classes::allClasses();

        return view("admin.level.edit", compact("level", "levels", "classes"));
    }

    /**
     * Update level with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public  function update(Request $request): RedirectResponse
    {
        $request->validate(LevelRules::update());

        $id = $request->{"id"};
        $designation = $request->{"designation"};
        $classIds = $request->{"classIds"};

        $response = Level::updateLevel($id, $designation, $classIds);

        if($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('level.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete level with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(LevelRules::delete());
        $id = $request->{"id"};

        $response = Level::deleteLevel($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Get levels with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getLevels(Request $request): JsonResponse
    {
        $request->validate(LevelRules::getLevels());

        $id = $request->{"education_type_id"};

        $EducationType = EducationType::findById($id);

        $levels = array();
        foreach ($EducationType->{"Levels"} as $level) {
            $levels[] =array(
                "id" => $level->{"id"},
                "designation" => $level->{"designation"}
            );
        }

        return response()->json($levels);
    }

}
