<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Models\Classes;
use App\Models\EducationType;
use App\Models\Level;
use App\Models\Specialty;
use App\RequestRules\Web\ClassRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class ClassController extends Controller
{

    /**
     * Get view
     *
     * @return Application|Factory|View
     */
    public function index(): View|Factory|Application
    {
        $classes = Classes::allClasses();

        return view("admin.class.index", compact('classes'));
    }

    /**
     * Create new class
     */
    public function create(Request $request): RedirectResponse
    {
        $request->validate(ClassRules::create());

        $designation = $request->{"designation"};

        $response = Classes::createClass($designation);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('class.index')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Return edit view
     *
     * @param Request $request
     * @return Factory|View|Application
     */
    public function edit(Request $request): Factory|View|Application
    {
        $request->validate(ClassRules::edit($request->merge(["id" => $request->route("id")])));
        $id = $request->{"id"};

        $class = Classes::findById($id);
        $classes = Classes::allClasses();

        return view("admin.class.edit", compact("class", "classes"));
    }

    /**
     * Update class with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(ClassRules::update());
        $id = $request->{"id"};
        $designation = $request->{"designation"};

        $response = Classes::updateClass($id, $designation);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('class.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete class with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(ClassRules::delete());

        $id = $request->{"id"};

        $response = Classes::deleteClass($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Get classes
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getClasses(Request $request): JsonResponse
    {
        $request->validate(ClassRules::getClasses());

        $id = $request->{"education_type_id"};

        $EducationType = EducationType::findById($id);

        $classes = array();
        foreach ($EducationType->{"classes"} as $class) {
            $classes[] = array(
                "id" => $class->{"id"},
                "designation" => $class->{"designation"}
            );
        }

        return response()->json($classes);
    }

    /**
     * Get specialty classes
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getSpecialtyClasses(Request $request): JsonResponse
    {
        $request->validate(ClassRules::getSpecialtyClasses());

        $id = $request->{"specialty_id"};

        $specialty = Specialty::findById($id);

        $specialtyClasses = array();
        foreach ($specialty->{"classes"} as $specialtyClass) {
            $specialtyClasses[] = array(
                "id" => $specialtyClass->{"id"},
                "designation" => $specialtyClass->{"designation"}
            );
        }

        return response()->json($specialtyClasses);
    }

    /**
     * Get level classes
     *
     * @param Request $request
     * @return JsonResponse
     */
    public  function getLevelClasses(Request $request): JsonResponse
    {
        $request->validate(ClassRules::getLevelClasses());

        $id = $request->{"level_id"};

        $levels = Level::findById($id);

        $levelClasses = array();
        foreach ($levels->{"classes"} as $levelClass) {
            $levelClasses[] = array(
                "id" => $levelClass->{"id"},
                "designation" => $levelClass->{"designation"}
            );
        }

        return response()->json($levelClasses);
    }
}
