<?php

namespace App\Services;

use App\Models\SubjectGroupClass;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;

trait SubjectGroupClassService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return SubjectGroupClass::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Create subject group for general education classes
     *
     * @param int $subjectGroupId
     * @param int $subSystemId
     * @param int $educationTypeId
     * @param int $educationTypeClassId
     * @return Builder|Model
     */
    public static function createSubjectGroupGeneralEducation(int $subjectGroupId, int $subSystemId, int $educationTypeId, int $educationTypeClassId): Model|Builder
    {
        return self::store([
            "subject_group_id" => $subjectGroupId,
            "sub_system_id" => $subSystemId,
            "education_type_id" => $educationTypeId,
            "sub_education_type_id" => 0,
            "education_type_class_id" => $educationTypeClassId
        ]);
    }

    /**
     * Create subject group for technical education classes
     *
     * @param int $subjectGroupId
     * @param int $subSystemId
     * @param int $educationTypeId
     * @param int $subEducationTypeId
     * @param int $specialtyClassId
     * @return Builder|Model
     */
    public static function createSubjectGroupTechnicalEducation(int $subjectGroupId, int $subSystemId, int $educationTypeId, int $subEducationTypeId, int $specialtyClassId): Model|Builder
    {
        return self::store([
            "subject_group_id" => $subjectGroupId,
            "sub_system_id" => $subSystemId,
            "education_type_id" => $educationTypeId,
            "sub_education_type_id" => $subEducationTypeId,
            "specialty_class_id" => $specialtyClassId
        ]);
    }

    /**
     * Create subject group for Teacher training classes
     *
     * @param int $subjectGroupId
     * @param int $subSystemId
     * @param int $educationTypeId
     * @param int $levelClassId
     * @return Builder|Model
     */
    public static function createSubjectGroupTeacherTraining(int $subjectGroupId, int $subSystemId, int $educationTypeId, int $levelClassId): Model|Builder
    {
        return self::store([
            "subject_group_id" => $subjectGroupId,
            "sub_system_id" => $subSystemId,
            "education_type_id" => $educationTypeId,
            "sub_education_type_id" => 0,
            "level_class_id" => $levelClassId
        ]);
    }

    /**
     * Find by subject group id
     *
     * @param int $subjectGroupId
     * @return Builder
     */
    public static function findBySubjectGroupId(int $subjectGroupId): Builder
    {
        return SubjectGroupClass::query()->where('subject_group_id', $subjectGroupId);
    }


    /**
     * Get subject group id
     *
     * @param string $type
     * @param int $id
     * @return Model|Builder|null
     */
    public static function getClassSubjectGroup(string $type, int $id): Model|Builder|null
    {
        $subjectGroupId = SubjectGroupClass::query();

        if($type == config('minesec.education_type.general')) {
            $subjectGroupId->whereNull(array('specialty_class_id', 'level_class_id'))
                ->where('education_type_class_id', $id);
        }

        if($type == config('minesec.education_type.technical')) {
            $subjectGroupId->whereNull(array('education_type_class_id', 'level_class_id'))
                ->where('specialty_class_id', $id);
        }

        if($type == config('minesec.education_type.teacher_training')) {
            $subjectGroupId->whereNull(array('education_type_class_id', 'specialty_class_id'))
                ->where('level_class_id', $id);
        }

        return $subjectGroupId->first();
    }

}
