<?php

namespace App\Services;

use App\Models\SubSystem;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use PHPUnit\Exception;

trait SubSystemService
{

    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return SubSystem::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all subsystems
     *
     * @return Builder[]|Collection
     */
    public static  function  allSubSystems(): Collection|array
    {
        return SubSystem::query()->get();
    }

    /**
     * Create sub system
     *
     * @param string $designation
     * @param array $educationTypeIds
     * @return SubSystem|Model|Builder|null
     */
    public  static  function createSubSystem(string $designation, Array $educationTypeIds): SubSystem|Model|Builder|null
    {

        DB::beginTransaction();

        try {
            /** @var SubSystem $result */

            $subSystem = self::store([
                "designation" => $designation,
            ]);

            if($subSystem != null) {
                $subSystem->educationTypes()->attach($educationTypeIds);
                $result = $subSystem;
                DB::commit();
            }

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubSystemService::createSubSystem');
            DB::rollBack();
            $result = null;
        }

        return $result;
    }

    /**
     * Find by given id
     *
     * @param int $id
     * @return Model|Builder|null
     */
    public static function findById(int $id): Model|Builder|null
    {
        return SubSystem::query()->where('id', $id)->first();
    }

    /**
     * Update sub system with given id
     *
     * @param int $id
     * @param string $designation
     * @param array $educationTypeIds
     * @return Model|Builder|null
     */
    public static function updateSubsystem(int $id, string $designation, Array $educationTypeIds): Model|Builder|null
    {
        DB::beginTransaction();
        try {
            SubSystem::query()->where('id', $id)->update(["designation" => $designation]);
            $subSystem = self::findById($id);
            if($subSystem != null) {
                $subSystem->educationTypes()->sync($educationTypeIds);
                DB::commit();
            }

        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubSystemService::updateSubsystem');
            DB::rollBack();
            $subSystem = null;
        }

        return $subSystem;
    }

    /**
     * Delete sub system with given id
     *
     * @param int $id
     * @return SubSystem|null
     */
    public static  function deleteSubSystem(int $id): ?SubSystem
    {
        DB::beginTransaction();
        try {
            /** @var SubSystem $subSystem */
            $subSystem = self::findById($id);

            if($subSystem != null) {
                $subSystem->educationTypes()->detach();
                $subSystem->delete();
                DB::commit();
            }

        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubSystemService::deleteSubSystem');
            DB::rollBack();
            $subSystem = null;
        }

        return $subSystem;
    }

    /**
     * Create sub  system subjects
     *
     * @param string $subSystemId
     * @param array $subjectIds
     * @return SubSystem|Exception
     */
    public static function createSubSystemSubjects(string $subSystemId, array $subjectIds): SubSystem|Exception
    {
        DB::beginTransaction();

        try {
            /** @var SubSystem $results */

            $subSystem = self::findById($subSystemId);

            $subSystem->subjects()->attach($subjectIds);
            $results = $subSystem;
            DB::commit();

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubSystemService::createSubSystemSubjects');
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Update sub system subjects with given id
     *
     * @param string $subSystemId
     * @param array $subjectIds
     * @return SubSystem|Exception
     */
    public static function updateSubSystemSubjects(string $subSystemId, array $subjectIds): SubSystem|Exception
    {
        DB::beginTransaction();

        try {
            /** @var SubSystem $results */

            $subSystem = self::findById($subSystemId);
            $subSystem->subjects()->sync($subjectIds);
            $results = $subSystem;
            DB::commit();

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubSystemService::updateSubSystemSubjects');
            DB::rollBack();
            $results = null;
        }

        return $results;
    }

    /**
     * Delete sub system subjects with given id
     *
     * @param string $subSystemId
     * @return SubSystem|Exception
     */
    public static function deleteSubSystemSubjects(string $subSystemId): SubSystem|Exception
    {
        DB::beginTransaction();

        try {
            /** @var SubSystem $subSystem */

            $subSystem = self::findById($subSystemId);
            $subSystem->subjects()->detach();
            DB::commit();

        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'SubSystemService::deleteSubSystemSubjects');
            DB::rollBack();
            $subSystem = null;
        }

        return $subSystem;
    }
}
