<?php

namespace App\Services;

use App\Models\Role;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

trait RoleService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return Role::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * Get all roles
     *
     * @return Builder[]|Collection
     */
    public static function allRoles(): Collection|array
    {
        return Role::query()->get();
    }

    /**
     * Create roles
     *
     * @param array $data
     * @return Builder|Model|null
     */
    public static function createRole(array $data): Model|Builder|null
    {
        try {
            $role = self::store([
                "role_name" => str_replace(' ', '_', strtolower($data["name"])),
                "name" => $data["name"],
                "guard_name" => "web",
                "is_system" => 0,
            ]);

            $results = $role;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'RoleService::createRole');
            $results = null;
        }

        return $results;
    }

    /**
     * Update role
     *
     * @param array $data
     * @return Role|null
     */
    public function updateRole(array $data): ?Role
    {
        try {
            $role = $this;

            $role->updateService([
                "role_name" => str_replace(' ', '_', strtolower($data["name"])),
                "name" => $data["name"],
            ]);

            $results = $role;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'RoleService::updateRole');
            $results = null;
        }

        return $results;
    }

    /**
     * Delete role
     *
     * @param int $id
     * @return int|null
     */
    public static function deleteRole(int $id): ?int
    {
        try {
            $role = Role::destroy($id);
        } catch (\Exception $exception) {
            log_debug(exception: $exception, prefix: 'RoleService::deleteRole');
            $role = null;
        }
        return $role;
    }

    /**
     * Find by role id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return Role::query()->find($id);
    }

    /**
     * Get all roles
     *
     * @return Builder[]|Collection
     */
    public static function getSystemRoles(): Collection|array
    {
        return Role::query()->where("is_system", config("minesec.role_types.system_roles"))->get();
    }

    /**
     * Get inspectorate roles
     *
     * @return Builder[]|Collection
     */
    public static function getInspectorateRoles(): Collection|array
    {
        return Role::query()->where("is_system", config("minesec.role_types.non_system_roles"))->get();
    }

    /**
     * @param array $data
     * @return \Spatie\Permission\Models\Role|\Spatie\Permission\Contracts\Role|null
     */
    public static function assignPermissionsToRole(array $data): \Spatie\Permission\Models\Role|null|\Spatie\Permission\Contracts\Role
    {
        try {
            $role = \Spatie\Permission\Models\Role::findById($data["role_id"]);

            $assignedPermissions = $role->syncPermissions($data["permissionIds"]);
            $result = $assignedPermissions;
        } catch (Exception $exception) {
            log_debug(exception: $exception, prefix: 'RoleService::assignPermissionsToRole');
            $result = null;
        }

        return $result;
    }

}
