<?php

namespace App\Services;

use App\Enums\ValidationStageEnum;
use App\Enums\ValidationStageStatus;
use App\Models\LessonProgressionDetail;
use App\Models\User;
use Exception;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;

trait LessonProgressionDetailsService
{
    /**
     * Add new record
     *
     * @param array $data
     * @return Builder|Model
     */
    public static function store(array $data): Model|Builder
    {
        return LessonProgressionDetail::query()->create($data);
    }

    /**
     * Update existing record
     *
     * @param array $data
     * @return mixed
     */
    public function updateService(array $data): mixed
    {
        return tap($this)->update($data);
    }

    /**
     * FInd by id
     *
     * @param int $id
     * @return Builder|Builder[]|Collection|Model|null
     */
    public static function findById(int $id): Model|Collection|Builder|array|null
    {
        return LessonProgressionDetail::query()->find($id);
    }

    /**
     * Find by lesson progression id
     *
     * @param int $lessonProgressionId
     * @return Builder[]|Collection
     */
    public static function findByLessonProgressionId(int $lessonProgressionId): Collection|array
    {
        return LessonProgressionDetail::query()->where("lesson_progression_id", $lessonProgressionId)->get();
    }

    /**
     * Create general education lesson progression details
     *
     * @param User $user
     * @param int $lessonProgressionId
     * @return Model|Builder|null
     */
    public static function submitGeneralEducationLessonDetails(User $user, int $lessonProgressionId): Model|Builder|null
    {
        try {
            $lessonProgressionDetails = LessonProgressionDetail::store([
                'stage' => ValidationStageEnum::submission->value,
                'status' => ValidationStageStatus::submissionPending->value,
                'lesson_progression_id' => $lessonProgressionId,
                'submitted_by' => $user->{"id"},
                'submitted_at' => now(),
                'submitter_phone_number' => $user->{"phone_number"},
            ]);

            $results = $lessonProgressionDetails;
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionDetailsService::submitGeneralEducationLessonDetails");
            $results = null;
        }

        return $results;
    }

    /**
     * Create technical education lesson progression details
     *
     * @param User $user
     * @param int $lessonProgressionId
     * @return Builder|Model|null
     */
    public static function submitTechnicalEducationLessonDetails(User $user, int $lessonProgressionId): Model|Builder|null
    {
        try {
            $lessonProgressionDetails = LessonProgressionDetail::store([
                'stage' => ValidationStageEnum::submission->value,
                'status' => ValidationStageStatus::submissionPending->value,
                'lesson_progression_id' => $lessonProgressionId,
                'submitted_by' => $user->{"id"},
                'submitted_at' => now(),
                'submitter_phone_number' => $user->{"phone_number"},
            ]);

            $results = $lessonProgressionDetails;
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionDetailsService::submitTechnicalEducationLessonDetails");
            $results = null;
        }

        return $results;
    }

    /**
     * Create teacher training lesson progression details
     *
     * @param $user
     * @param int $lessonProgressionId
     * @return Builder|Model|null
     */
    public static function submitTeacherTrainingLessonDetails($user, int $lessonProgressionId): Model|Builder|null
    {
        try {
            $lessonProgressionDetails = LessonProgressionDetail::store([
                'stage' => ValidationStageEnum::submission->value,
                'status' => ValidationStageStatus::submissionPending->value,
                'lesson_progression_id' => $lessonProgressionId,
                'submitted_by' => $user->{"id"},
                'submitted_at' => now(),
                'submitter_phone_number' => $user->{"phone_number"},
                ]);

            $results = $lessonProgressionDetails;
        } catch (Exception $exception) {
            log_debug($exception, "LessonProgressionDetailsService::submitTeacherTrainingLessonDetails");
            $results = null;
        }

        return $results;
    }

}
