<?php

namespace App\Http\Controllers\Web\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\Auth;

class AuthenticationController extends Controller
{

    /**
     *  Get authentication view
     *
     * @return Application|Factory|View
     */
    public function auth(): View|Factory|Application
    {
        return view("auth.login");
    }

    /**
     * Handle an authentication attempt.
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function authentication(Request $request): RedirectResponse
    {
        $credentials = $request->validate([
            'phone_number' => ['required'],
            'password' => ['required'],
        ]);


        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();

            /** @var User $user */
            $user = auth()->user();

            // Set language locale
            User::setLanguage($user->{"locale"});

            if($user->hasRole(config("minesec.roles.teacher")) || $user->hasRole(config("minesec.roles.ICG")) || $user->hasRole(config("minesec.roles.CDS")) || $user->hasRole(config("minesec.roles.IPN")) || $user->hasRole(config("minesec.roles.ICR")) || $user->hasRole(config("minesec.roles.IPR"))) {

                //Put inspectorate id in the session
                $request->session()->put('inspectorate_id', $user->{"inspectorate_id"});

                return redirect()->intended("inspectorate/dashboard");
            }

            return redirect()->intended('admin/dashboard');
        }

        return back()->withErrors(["error" => __("error.invalid_credentials")]);
    }

    /**
     * Log the user out of the application.
     *
     * @param Request $request
     * @return Application|Redirector|RedirectResponse
     */
    public function logout(Request $request): Redirector|RedirectResponse|Application
    {
        Auth::logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect("login");
    }
}
