<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Imports\SchoolImport;
use App\Models\Region;
use App\Models\School;
use App\RequestRules\Web\SchoolRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;
use Maatwebsite\Excel\Facades\Excel;


class SchoolController extends Controller
{
    /**
     * Import schools
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function import(Request $request): RedirectResponse
    {
        $request->validate(SchoolRules::import());

        Excel::import(new SchoolImport, $request->file('schools'));

        return Redirect::route('schools.index')->with("success", __("messages.record_successfully_imported"));
    }

    /**
     * Get schools view
     *
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function index(): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $schools = School::allSchools();
        $regions = Region::allRegions();

        return view("admin.school.index", compact("schools", "regions"));
    }

    /**
     * Create school
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {
        $request->validate(SchoolRules::createRules());
        $data = $request->all();

        $results = School::createSchool($data);

        if ($results == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('schools.index')->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Edit view
     *
     * @param Request $request
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(Request $request): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $request->validate(SchoolRules::edit($request->merge(["id" => $request->route("id")])));

        $schools = School::allSchools();
        $regions = Region::allRegions();
        $schoolDetail = School::findById($request->{"id"});

        return view("admin.school.edit", compact("schools","schoolDetail", "regions"));
    }

    /**
     * Update school
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(SchoolRules::update());
        $data = $request->all();

        /** @var School $SchoolToUpdate */
        $SchoolToUpdate = School::findById($data["school_id"]);

        if($SchoolToUpdate == null) {
            return Redirect::back()->with("error", __("error.unknown_school"));
        }

        $results = $SchoolToUpdate->updateSchool($data);

        if ($results == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('schools.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete school
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(SchoolRules::deleteSchool());

        $response = School::deleteSchool($request->{"id"});

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }


}
