<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Models\PermissionGroup;
use App\Models\Role;
use App\RequestRules\Web\RolePermissionRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class RolePermissionController extends Controller
{
    /**
     * Get roles index view
     *
     * @return Application|Factory|View
     */
    public function index(): View|Factory|Application
    {
        $roles = Role::allRoles();

        return view("admin.settings.role_permission.index", compact('roles'));
    }

    /**
     * Create role
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {
        $request->validate(RolePermissionRules::createRole());
        $data = $request->all();

        $response = Role::createRole($data);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::back()->with("success", __("messages.record_successfully_created"));
    }

    /**
     * Edit role view
     *
     * @param Request $request
     * @return Application|Factory|View|\Illuminate\Foundation\Application
     */
    public function edit(Request $request): \Illuminate\Foundation\Application|View|Factory|Application
    {
        $request->validate(RolePermissionRules::edit($request->merge(["id" => $request->route("id")])));
        $roles = Role::allRoles();
        $roleToEdit = Role::findById($request->{"id"});

        return view("admin.settings.role_permission.edit", compact("roles", "roleToEdit"));
    }

    /**
     * Update role
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        $request->validate(RolePermissionRules::update());
        $data = $request->all();

        /** @var Role $role */
        $role = Role::findById($data["role_id"]);

        if($role == null) {
            return Redirect::back()->with("error", __("error.unknown_role"));
        }

        $response = $role->updateService($data);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('role.index')->with("success", __("messages.record_successfully_updated"));
    }

    /**
     * Delete role
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        $request->validate(RolePermissionRules::delete());

        $response = Role::deleteRole($request->{"id"});

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.record_successfully_deleted")]);
    }

    /**
     * Get permissions view
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function rolePermission(Request $request): View|Factory|Application
    {
        $request->validate(RolePermissionRules::getPermissions($request->merge(["id" => $request->route("id")])));

        $role = Role::findById($request->get("id"));
        $permissionGroups = PermissionGroup::allPermissionGroups();
        $roleHasPermissions = $role->{"roleHasPermission"};

        return view("admin.settings.role_permission.permission", compact("permissionGroups","role", "roleHasPermissions"));
    }

    /**
     * Assign permissions to role
     * @param Request $request
     * @return RedirectResponse
     */
    public function assign(Request $request): RedirectResponse
    {
        $request->validate(RolePermissionRules::assignPermissionsToRoles());

       $data = $request->all();

        $response = Role::assignPermissionsToRole($data);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::back()->with("success", __("messages.record_successfully_created"));
    }

}
