<?php

namespace App\Http\Controllers\Web\Admin;

use App\Enums\ResourceUploadTypeEnum;
use App\Http\Controllers\Controller;
use App\Imports\GeneralEducationLessonsImport;
use App\Imports\TeacherTrainingLessonsImport;
use App\Imports\TechnicalEducationLessonsImport;
use App\Libraries\ConstantsLib;
use App\Models\Classes;
use App\Models\EducationType;
use App\Models\Inspectorate;
use App\Models\Lesson;
use App\Models\LessonProgression;
use App\Models\Subject;
use App\Models\SubSystem;
use App\Models\Term;
use App\RequestRules\Web\LessonProgressionRules;
use App\RequestRules\Web\LessonRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\Routing\ResponseFactory;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Redirect;
use Maatwebsite\Excel\Facades\Excel;

class LessonController extends Controller
{

    /**
     * @var ConstantsLib
     */
    protected ConstantsLib $ConstantsLib;

    public function __construct(ConstantsLib $ConstantsLib)
    {
        $this->ConstantsLib = $ConstantsLib;
    }

    /**
     * Get add lesson view
     *
     * @return Application|Factory|View
     */
    public function index(): View|Factory|Application
    {
        $subSystems = SubSystem::allSubSystems();
        $classes = Classes::allClasses();
        $terms = Term::allTerms();
        $subjects = Subject::allSubjects();
        $lessons = Lesson::allLessons();
        $resourceDestination = $this->ConstantsLib->getResourceDestination();

        return view("admin.lesson.index", compact("subSystems", "classes", "terms", "subjects", "lessons", "resourceDestination"));
    }

    /**
     * Create lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function create(Request $request): JsonResponse
    {
        $data = $request->all();

        $educationTypeId = preg_replace("/[^0-9]/", "", $data['education_type_id']);
        $type = EducationType::getEducationType((int)$educationTypeId);

        if ($type != null && $type == config('minesec.education_type.general')) {
            $request->validate(LessonRules::createGeneralEducationLesson());
            if (isset($data["upload_file"])) {
                $request->session()->now('type', $type);
                $request->session()->now('sub_system_id', $data["sub_system_id"]);
                $request->session()->now('education_type_id', $educationTypeId);
                $request->session()->now('class_id', $data["class_id"]);
                $request->session()->now('subject_id', $data['subject_id']);
                $request->session()->now('term_id', $data["term_id"]);

                Excel::import(new GeneralEducationLessonsImport, $request->file('lessons'));
                return response()->json(["success" => __("messages.record_successfully_imported")]);
            }

            $response = Lesson::createLessonGeneralEducation($data, $educationTypeId, $type);
            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_created")]);
        }

        if ($type != null && $type == config('minesec.education_type.technical')) {
            $request->validate(LessonRules::createTechnicalEducationLesson());
            if (isset($data["upload_file"])) {
                $request->session()->now('type', $type);
                $request->session()->now('sub_system_id', $data["sub_system_id"]);
                $request->session()->now('education_type_id', $educationTypeId);
                $request->session()->now('sub_education_type_id', $data["sub_education_type_id"]);
                $request->session()->now('specialty_id', $data["specialty_id"]);
                $request->session()->now('class_id', $data["specialty_class_id"]);
                $request->session()->now('subject_id', $data['subject_id']);
                $request->session()->now('term_id', $data["term_id"]);

                Excel::import(new TechnicalEducationLessonsImport, $request->file('lessons'));
                return response()->json(["success" => __("messages.record_successfully_imported")]);
            }

            $response = Lesson::createLessonTechnicalEducation($data, $educationTypeId, $type);
            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_created")]);
        }

        if ($type != null && $type == config('minesec.education_type.teacher_training')) {
            $request->validate(LessonRules::createTeacherTrainingLesson());
            if (isset($data["upload_file"])) {
                $request->session()->now('type', $type);
                $request->session()->now('sub_system_id', $data["sub_system_id"]);
                $request->session()->now('education_type_id', $educationTypeId);
                $request->session()->now('level_id', $data["level_id"]);
                $request->session()->now('class_id', $data["level_class_id"]);
                $request->session()->now('subject_id', $data['subject_id']);
                $request->session()->now('term_id', $data["term_id"]);

                Excel::import(new TeacherTrainingLessonsImport, $request->file('lessons'));
                return response()->json(["success" => __("messages.record_successfully_imported")]);
            }

            $response = Lesson::createLessonTeacherTraining($data, $educationTypeId, $type);
            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_created")]);
        }

        return response()->json(["error" => __("error.generic_error")]);
    }

    public function edit(Request $request)
    {

        dd($request->merge(["id" => $request->route("id")]));
    }

    /**
     * Delete lesson with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function delete(Request $request): RedirectResponse
    {
        $request->validate(LessonRules::delete($request->merge(["id" => $request->route("id")])));

        $id = $request->{"id"};

        $response = Lesson::deleteLesson($id);

        if ($response == null) {
            return Redirect::back()->with("error", __("error.generic_error"));
        }

        return Redirect::route('lesson.index')->with("success", __("messages.record_successfully_deleted"));
    }

    /**
     * Get lessons to be validared
     *
     * @return Application|Factory|View
     */
    public function lessons(): View|Factory|Application
    {
        $subSystems = SubSystem::allSubSystems();
        $classes = Classes::allClasses();
        $subjects = Subject::allSubjects();
        $inspectorates = Inspectorate::allInspectorates();
        $lessonProgressions = LessonProgression::allLessonProgression();

        return view("admin.lesson.validation.index", compact("subSystems", "classes", "subjects", "lessonProgressions", "inspectorates"));
    }

//    /**
//     * Submit lesson for validation
//     *
//     * @param Request $request
//     * @return JsonResponse
//     */
//    public function addLesson(Request $request): JsonResponse
//    {
//        $data = $request->all();
//
//        $educationTypeId = preg_replace("/[^0-9]/", "", $data['education_type_id']);
//        $type = EducationType::getEducationType((int)$educationTypeId);
//
//        if ($type != null && $type == config('minesec.education_type.general')) {
//            $request->validate(LessonProgressionRules::submitGeneralEducationLesson());
//            $response = LessonProgression::submitGeneralEducationLesson($data, $educationTypeId, $type);
//
//            if ($response == null) {
//                return response()->json(["error" => __("error.generic_error")]);
//            }
//
//            return response()->json(["success" => __("messages.record_successfully_created")]);
//        }
//
//        if ($type != null && $type == config('minesec.education_type.technical')) {
//            $request->validate(LessonProgressionRules::submitTechnicalEducationLesson());
//
//            $response = LessonProgression::submitTechnicalEducationLesson($data, $educationTypeId, $type);
//            if ($response == null) {
//                return response()->json(["error" => __("error.generic_error")]);
//            }
//
//            return response()->json(["success" => __("messages.record_successfully_created")]);
//        }
//
//        if ($type != null && $type == config('minesec.education_type.teacher_training')) {
//            $request->validate(LessonProgressionRules::submitTeacherTrainingLesson());
//            $response = LessonProgression::submitTeacherTrainingLesson($data, $educationTypeId, $type);
//
//            if ($response == null) {
//                return response()->json(["error" => __("error.generic_error")]);
//            }
//
//            return response()->json(["success" => __("messages.record_successfully_created")]);
//        }
//
//        return response()->json(["error" => __("error.generic_error")]);
//    }

    /**
     * Show lesson details
     *
     * @param Request $request
     * @return View|Factory|Application
     */
    public function showLesson(Request $request): View|Factory|Application
    {
        $request->validate(LessonProgressionRules::showLesson($request->merge(["id" => $request->route("id")])));

        $lessonDetails = LessonProgression::findById($request->get("id"));

        return view("admin.lesson.validation.details", compact("lessonDetails"));
    }

    /**
     * Approve lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function approveSubmission(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::approveSubmission());

        $data = $request->all();

        $response = LessonProgression::approveSubmission($data);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_approved")]);
    }

    /**
     * Reject lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public static function rejectSubmission(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::rejectSubmission());

        $data = $request->all();

        $response = LessonProgression::rejectSubmission($data);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_rejected")]);
    }

    /**
     * Get approved lesson
     *
     * @return Application|Factory|View
     */
    public function approvedLessons(): View|Factory|Application
    {
        $lessonProgressions = LessonProgression::validatedLessons();

        return view("admin.lesson.recording.index", compact("lessonProgressions"));
    }

    /**
     * Show validated lesson details
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public static function showApprovedLesson(Request $request): View|Factory|Application
    {
        $request->validate(LessonProgressionRules::showLesson($request->merge(["id" => $request->route("id")])));

        $lessonDetails = LessonProgression::findById($request->get("id"));

        return view("admin.lesson.recording.details", compact("lessonDetails"));
    }

    /**
     * Approve validates lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public static function approveValidatedLesson(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::approveRejectLesson());

        $id = $request->{"lesson_progression_id"};

        $response = LessonProgression::approveValidatedLesson($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_approved")]);
    }

    /**
     * Reject validated lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public static function rejectedValidatedLesson(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::approveRejectLesson());
        $data = $request->all();

        $response = LessonProgression::rejectValidatedLesson($data);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_rejected")]);
    }

    /**
     * Get recorde lessons
     *
     * @return Application|Factory|View
     */
    public function recordedLessons(): View|Factory|Application
    {
        $lessonProgressions = LessonProgression::recordedLessons();

        return view("admin.lesson.editing.index", compact("lessonProgressions"));
    }

    /**
     * Show recorded lesson details
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function showRecordedLesson(Request $request): View|Factory|Application
    {
        $request->validate(LessonProgressionRules::showLesson($request->merge(["id" => $request->route("id")])));

        $lessonDetails = LessonProgression::findById($request->get("id"));

        return view("admin.lesson.editing.details", compact("lessonDetails"));
    }

    /**
     * Approve recorded lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function approveRecordedLesson(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::approveRejectLesson());

        $id = $request->{"lesson_progression_id"};

        $response = LessonProgression::approveRecordedLesson($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_approved")]);
    }

    /**
     * Reject recorded lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function rejectedRecordedLesson(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::approveRejectLesson());

        $data = $request->all();

        $response = LessonProgression::rejectRecordedLesson($data);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_rejected")]);
    }

    /**
     * Get edited lessons
     *
     * @return Application|Factory|View
     */
    public function editedLessons(): View|Factory|Application
    {
        $lessonProgressions = LessonProgression::editedLessons();

        return view("admin.lesson.segmenting.index", compact("lessonProgressions"));
    }

    /**
     * Show edited lesson details
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function showEditedLesson(Request $request): View|Factory|Application
    {
        $request->validate(LessonProgressionRules::showLesson($request->merge(["id" => $request->route("id")])));

        $lessonDetails = LessonProgression::findById($request->get("id"));

        return view("admin.lesson.segmenting.details", compact("lessonDetails"));
    }

    /**
     * Approve edited lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function approveEditedLesson(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::approveRejectLesson());

        $id = $request->{"lesson_progression_id"};

        $response = LessonProgression::approveEditedLesson($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_approved")]);
    }

    /**
     * Reject edited lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function rejectedEditedLesson(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::approveRejectLesson());

        $data = $request->all();

        $response = LessonProgression::rejectedEditedLesson($data);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_rejected")]);
    }

    /**
     * Get segmented lessons
     *
     * @return Application|Factory|View
     */
    public function segmentedLessons(): View|Factory|Application
    {
        $lessonProgressions = LessonProgression::segmentedLessons();

        return view("admin.lesson.put_in_class.index", compact("lessonProgressions"));
    }

    /**
     * Get segmented lessons
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function showSegmentedLesson(Request $request): View|Factory|Application
    {
        $request->validate(LessonProgressionRules::showLesson($request->merge(["id" => $request->route("id")])));

        $lessonDetails = LessonProgression::findById($request->get("id"));

        return view("admin.lesson.put_in_class.details", compact("lessonDetails"));
    }

    /**
     * Put segmented lesson in class
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function putSegmentedLessonInClass(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::approveRejectLesson());

        $id = $request->{"lesson_progression_id"};

        $response = LessonProgression::putSegmentedLessonInClass($id);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_approved")]);
    }

    /**
     * Reject segmented lesson
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function rejectSegmentedLesson(Request $request): JsonResponse
    {
        $request->validate(LessonProgressionRules::approveRejectLesson());

        $data = $request->all();

        $response = LessonProgression::rejectSegmentedLesson($data);

        if ($response == null) {
            return response()->json(["error" => __("error.generic_error")]);
        }

        return response()->json(["success" => __("messages.lesson_successfully_rejected")]);
    }

    /**
     * Get class lesson
     *
     * @param Request $request
     * @return Response|Application|ResponseFactory
     */
    public function getSubjectLessons(Request $request): Response|Application|ResponseFactory
    {
        $data = $request->all();
        $response = Lesson::subjectLessons($data);

        if (count($response) == 0) {
            return api_response(150, 'No lessons found');
        }

        $lessonArray = array();
        foreach ($response as $lesson) {
            $lessonArray[] = array(
                "id" => $lesson->{"id"},
                "title" => $lesson->{"lessonTitle"}->{"title"},
                "description" => $lesson->{"lessonTitle"}->{"description"},
                "lesson_image" => url('storage/' . $lesson->{"thumbnail"}),
                "path" => $lesson->{"upload_type"} == ResourceUploadTypeEnum::localServer->value ?  url('storage/' . $lesson->{"path"}) : $lesson->{"path"},
                "duration" => $lesson->{"duration"},
                "upload_type" => $lesson->{"upload_type"},
//                "path" =>  URL::to('/'.$lesson->{"path"}),
//                "visibility" => Storage::getVisibility($lesson->{"path"}),
            );
        }

        return api_response(100, 'ok', $lessonArray);
    }
}
