<?php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Controller;
use App\Models\Classes;
use App\Models\EducationType;
use App\Models\Level;
use App\Models\SubEducationType;
use App\Models\SubSystem;
use App\RequestRules\Web\EducationTypeRules;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Contracts\View\View;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Redirect;

class EducationTypeController extends Controller
{
    /**
     * Get education type view
     *
     * @return Application|Factory|View
     */
    public function index(): View|Factory|Application
    {
        $educationTypes = EducationType::allEducationTypes();
        $subEducationTypes = SubEducationType::allSubEducationTypes();
        $levels = Level::allLevels();
        $classes = Classes::allClasses();

        return view("admin.education_type.index", compact("educationTypes", "subEducationTypes", "levels", "classes"));
    }

    /**
     * create education type
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function create(Request $request): RedirectResponse
    {
        $requestType = isset($request->{"type"}) ? config('minesec.education_type.technical') : config('minesec.education_type.general');

        $requestType1 = isset($request->{"type_1"}) ? config('minesec.education_type.teacher_training') : null;

        if ($requestType1 == null && $requestType == config('minesec.education_type.technical')) {
            $request->validate(EducationTypeRules::createForTechnical());

            $designation = $request->{"designation"};
            $subEducationTypeIds = $request->{"subEducationTypeIds"};
            $type = $requestType;

            $response = EducationType::createTechnicalEducation($designation, $subEducationTypeIds, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }
            return Redirect::route('education-type.index')->with("success", __("messages.record_successfully_created"));
        }

        if($requestType1 == null &&  $requestType == config('minesec.education_type.general')) {
            $request->validate(EducationTypeRules::createForGeneral());

            $designation = $request->{"designation"};
            $classIds = $request->{"classIds"};
            $type = $requestType;

            $response = EducationType::createGeneralEducation($designation, $classIds, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('education-type.index')->with("success", __("messages.record_successfully_created"));
        }

        if($requestType1 == config('minesec.education_type.teacher_training')) {
            $request->validate(EducationTypeRules::createForTeacherTraining());

            $designation = $request->{"designation"};
            $levelIds = $request->{"levelIds"};
            $type = $requestType1;

            $response = EducationType::createTeacherTraining($designation,$levelIds, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('education-type.index')->with("success", __("messages.record_successfully_created"));
        }

        return Redirect::back()->with("error", __("error.generic_error"));
    }

    /**
     * Get education type with given id
     *
     * @param Request $request
     * @return Application|Factory|View
     */
    public function edit(Request $request): View|Factory|Application
    {
        $request->validate(EducationTypeRules::edit($request->merge(["id" => $request->route("id")])));
        $id = $request->{"id"};

        $educationTypes = EducationType::allEducationTypes();
        $subEducationTypes = SubEducationType::allSubEducationTypes();
        $levels = Level::allLevels();
        $classes = Classes::allClasses();
        $educationType = EducationType::findById($id);

        return view("admin.education_type.edit", compact("educationTypes", "subEducationTypes", "levels", "classes", "educationType"));
    }

    /**
     * Update education type with given id
     *
     * @param Request $request
     * @return RedirectResponse
     */
    public function update(Request $request): RedirectResponse
    {
        if ($request->{"type"} == config('minesec.education_type.technical')) {
            $request->validate(EducationTypeRules::updateForTechnical());

            $id = $request->{"id"};
            $designation = $request->{"designation"};
            $subEducationTypeIds = $request->{"subEducationTypeIds"};
            $type = $request->{"type"};

            $response = EducationType::updateTechnicalEducation($id, $designation, $subEducationTypeIds, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('education-type.index')->with("success", __("messages.record_successfully_updated"));
        }

        if($request->{"type"} == config('minesec.education_type.general')) {
            $request->validate(EducationTypeRules::updateForGeneral());

            $id = $request->{"id"};
            $designation = $request->{"designation"};
            $classIds = $request->{"classIds"};
            $type = $request->{"type"};

            $response = EducationType::updateGeneralEducation($id, $designation, $classIds, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('education-type.index')->with("success", __("messages.record_successfully_updated"));
        }

        if($request->{"type"} == config('minesec.education_type.teacher_training')) {
            $request->validate(EducationTypeRules::updateForTeacherTraining());

            $id = $request->{"id"};
            $designation = $request->{"designation"};
            $levelIds = $request->{"levelIds"};
            $type = $request->{"type"};

            $response = EducationType::updateTeacherTrainingEducation($id, $designation, $levelIds, $type);

            if ($response == null) {
                return Redirect::back()->with("error", __("error.generic_error"));
            }

            return Redirect::route('education-type.index')->with("success", __("messages.record_successfully_updated"));
        }

        return Redirect::back()->with("error", __("error.generic_error"));
    }

    /**
     * Delete education type with given id
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function delete(Request $request): JsonResponse
    {
        if ($request->{"type"} == config('minesec.education_type.technical')) {
            $request->validate(EducationTypeRules::delete());
            $id = $request->{"id"};

            $response = EducationType::deleteTechnicalEducation($id);

            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_deleted")]);
        }

        if($request->{"type"} == config('minesec.education_type.general')){
            $request->validate(EducationTypeRules::delete());
            $id = $request->{"id"};

            $response = EducationType::deleteGeneralEducation($id);

            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_deleted")]);
        }

        if($request->{"type"} == config('minesec.education_type.teacher_training')) {
            $request->validate(EducationTypeRules::delete());
            $id = $request->{"id"};

            $response = EducationType::deleteTeacherTraining($id);

            if ($response == null) {
                return response()->json(["error" => __("error.generic_error")]);
            }

            return response()->json(["success" => __("messages.record_successfully_deleted")]);
        }

        return response()->json(["error" => __("error.generic_error")]);
    }


    /**
     * Get education types
     *
     * @param Request $request
     * @return JsonResponse
     */
    public function getEducationTypes(Request $request): JsonResponse
    {
        $request->validate(EducationTypeRules::getEducationTypes());

        $id = $request->{"sub_system_id"};

        $subSystem = SubSystem::findById($id);

        $educationTypes = array();
        foreach ($subSystem->{"educationTypes"} as $educationType) {
            $educationTypes[] =array(
                "id" => $educationType->{"id"},
                "type" => $educationType->{"type"},
                "designation" => $educationType->{"designation"}
            );
        }

        return response()->json($educationTypes);
    }
}
